package org.gridkit.quickrun.exec;

import java.util.concurrent.TimeUnit;

import org.gridkit.quickrun.exec.TaskSet.TaskWrapper;

/**
 * Adapts task sequence to fixed rate by adding condition.
 * <p>
 * This adapter implements open load model, so task could pile up if completion rate is below target rate.
 * It could be combined with
 */
public class FixedRateGate implements TaskWrapper {

    public static FixedRateGate atFreq(double freq) {
        long periodNs = (long) (TimeUnit.SECONDS.toNanos(1) / freq);
        if (periodNs <= 0) {
            throw new IllegalArgumentException("Invalid frequency: " + freq);
        }
        return new FixedRateGate(periodNs);
    }

    public static FixedRateGate ofMillis(long periodMs) {
        return new FixedRateGate(TimeUnit.MILLISECONDS.toNanos(periodMs));
    }

    public static FixedRateGate ofNanos(long periodNs) {
        return new FixedRateGate(periodNs);
    }

    private final long periodNS;
    private long nextTime;

    public FixedRateGate(long periodNS) {
        this.nextTime = System.nanoTime();
        this.periodNS = periodNS;
    }

    @Override
    public synchronized Task wrap(Task task) {
        DeadlineCondition cond = new DeadlineCondition(nextTime);
        nextTime += periodNS;
        SelectableLatch taskCond = cond.combine(task.condition());

        return new ProxyTask(task) {
            @Override
            public SelectableLatch condition() {
                return taskCond;
            }
        };
    }
}
