package org.gridkit.quickrun.exec;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Deque;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

public class LatchSelector {

    private final List<SelectableLatch> pendingLatches = new ArrayList<SelectableLatch>();

    private final Deque<SelectableLatch> openLatches = new ArrayDeque<SelectableLatch>();

    private Consumer<SelectableLatch> watch;

    public synchronized void add(SelectableLatch latch) {
        if (latch.isOpen()) {
            openLatches.add(latch);
        } else {
            pendingLatches.add(latch);
            watch = this::onOpen;
            latch.watch(watch);
        }
    }

    public synchronized SelectableLatch poll(long timeout, TimeUnit tu) throws InterruptedException {
        long deadline = System.nanoTime() + tu.toNanos(timeout);
        while(openLatches.isEmpty()) {
            waitForLatches(deadline);
            if (deadline < System.nanoTime()) {
                break;
            }
        }
        if (openLatches.isEmpty()) {
            return null;
        } else {
            return openLatches.removeFirst();
        }
    }

    private synchronized void waitForLatches(long deadline) throws InterruptedException {
        long sleepDelay = Long.MAX_VALUE;
        Iterator<SelectableLatch> it = pendingLatches.iterator();
        while(it.hasNext()) {

            SelectableLatch latch = it.next();
            long delay = latch.proposeWaitTimeNS();
            if (latch.isOpen()) {
                it.remove();
                openLatches.add(latch);
            } else {
                if (delay > 0) {
                    sleepDelay = Math.min(delay, sleepDelay);
                }
            }
        }

        if (openLatches.size() > 0) {
            return;
        } else {
            long stime = Math.min(sleepDelay, Math.max(0, deadline - System.nanoTime()));
            long stimeMS = TimeUnit.NANOSECONDS.toMillis(stime);
            if (stimeMS > 0) {
                this.wait(stimeMS);
            }
        }
    }

    private synchronized void onOpen(SelectableLatch latch) {
        if (latch.isOpen()) {
            pendingLatches.remove(latch);
            openLatches.add(latch);
            notifyAll();
        }
    }

    public synchronized void clean() {
        for (SelectableLatch pl: pendingLatches) {
            pl.unwatch(watch);
        }
        pendingLatches.clear();
        openLatches.clear();
    }
}
