package org.gridkit.quickrun.exec;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Consumer;

public class MultiLatch implements SelectableLatch, Consumer<SelectableLatch> {

    private final SelectableLatch a;
    private final SelectableLatch b;

    private final List<Consumer<SelectableLatch>> watchers = new ArrayList<Consumer<SelectableLatch>>();

    public MultiLatch(SelectableLatch a, SelectableLatch b) {
        this.a = a;
        this.b = b;
    }

    @Override
    public boolean isOpen() {
        return a.isOpen() && b.isOpen();
    }

    @Override
    public long proposeWaitTimeNS() {
        long ta = a.proposeWaitTimeNS();
        long tb = b.proposeWaitTimeNS();
        if (ta == 0) {
            return tb;
        } else if (tb == 0) {
            return ta;
        } else {
            return Math.min(ta, tb);
        }
    }

    @Override
    public void watch(Consumer<SelectableLatch> watch) {
        synchronized (this) {
            if (!isOpen()) {
                watchers.add(watch);
                return;
            }
        }
        watch.accept(this);
    }

    @Override
    public synchronized void unwatch(Consumer<SelectableLatch> watch) {
        watchers.remove(this);
    }

    @Override
    public void accept(SelectableLatch t) {
        List<Consumer<SelectableLatch>> toNotify = null;
        synchronized (this) {
            if (isOpen()) {
                toNotify = new ArrayList<>(watchers);
                watchers.clear();
            }
        }
        if (toNotify != null) {
            for (Consumer<SelectableLatch> watch: toNotify) {
                try {
                    watch.accept(this);
                } catch (Error e) {
                    throw e;
                } catch (Exception e) {
                    // ignore exceptions
                }
            }
        }
    }
}
