package org.gridkit.quickrun.exec;

import java.util.function.Supplier;

/**
 * Some, potentially bounded, ordered pool of task to execute.
 *
 * @author Alexey Ragozin (alexey.ragozin@gmail.com)
 */
public interface TaskSet {

    public static TaskSet of(final Runnable task) {
        return new TaskSet() {

            @Override
            public Task pollNextTask() {
                return Task.of(task);
            }
        };
    }

    public static TaskSet of(Supplier<Runnable> tasks) {
        return new TaskSet() {

            volatile boolean exhausted = false;

            @Override
            public Task pollNextTask() {
                Runnable task = tasks.get();
                if (task == null) {
                    exhausted = true;
                    return null;
                } else {
                    return Task.of(task);
                }
            }

            @Override
            public boolean isEmpty() {
                return exhausted;
            }
        };
    }

    public default boolean isEmpty() {
        return false;
    }

    /**
     * @return <code>null</code> if no more tasks available
     */
    public Task pollNextTask();

    /**
     * @return <code>null</code> if size estimation is not available
     */
    public default Long getRemaingSize() {
        return null;
    }

    public default TaskSet wrap(TaskWrapper wrapper) {
        final TaskSet delegate = this;
        return new TaskSet() {

            // switches to true once null task is encountered
            volatile boolean exhausted = false;

            @Override
            public Task pollNextTask() {
                Task task = delegate.pollNextTask();
                task = task == null ? null : wrapper.wrap(task);
                if (task == null) {
                    exhausted = true;
                }
                return task;
            }

            @Override
            public boolean isEmpty() {
                return exhausted || delegate.isEmpty();
            }

            @Override
            public Long getRemaingSize() {
                return delegate.getRemaingSize();
            }
        };
    }

    /**
     * Adds a gate to ensure at most one instance of task produced by this
     * task can run at any moment.
     */
    public default TaskSet inSequence() {
        return wrap(new ConcurrencyGate(1));
    }

    public default TaskSet limit(int maxTaskCount) {
        return wrap(new TaskCountGate(maxTaskCount));
    }

    public interface TaskWrapper {

        public Task wrap(Task task);

    }
}
