package org.gridkit.quickrun.exec;

import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executor;

/**
 * Simple scheduler strategy which just sequentially pull tasks and submit them.
 * Though if task is blocked on condition, no further tasks will be pulled.
 * <p>
 * This strategy could be combined with {@link ConcurrencyGate} and other gates to apply desired
 * concurrency or rate control.
 *
 * @author Alexey Ragozin (alexey.ragozin@gmail.com)
 *
 */
public class ThroughputScheduler implements Job {

    private Executor executor;
    private boolean started = false;
    private volatile boolean terminated = false;
    private CountDownLatch joinLatch = new CountDownLatch(1);

    private final TaskSet taskSet;
    private final ActiveTaskCounter taskCounter = new ActiveTaskCounter();

    public ThroughputScheduler(TaskSet taskSet) {
        this.taskSet = taskSet;
    }

    @Override
    public synchronized void start(Executor executor) {
        if (started) {
            throw new IllegalArgumentException("Job could be started only once");
        }
        this.executor = executor;
        executor.execute(this::schedule);
    }

    @Override
    public boolean isCompleted() {
        return (terminated || taskSet.isEmpty()) && taskCounter.getActiveTaskCount() == 0;
    }

    @Override
    public void stop() {
        terminated = true;
    }

    @Override
    public void join() throws InterruptedException {
        joinLatch.await();
        taskCounter.joinWithTasks();
    }

    private void schedule() {
        try {
            while (!terminated) {
                Task task = taskSet.pollNextTask();
                if (task == null) {
                    break;
                }
                Task wtask = taskCounter.wrap(task);
                SelectableLatch latch = wtask.condition();
                if (latch != null) {
                    latch.await();
                }
                executor.execute(() -> start(wtask));
            }
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        } finally {
            joinLatch.countDown();
        }
    }

    private void start(Task task) {
        if (!terminated) {
            try {
                task.start(executor);
            } catch (Exception e) {
                // ignore
            }
        }
    }
}
