package org.gridkit.quickrun.report;

import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

public class AsyncSampleSink implements Consumer<SampleRow>{

    private final Thread thread;
    private final BlockingQueue<SampleRow> queue;
    private final Consumer<SampleRow> sink;
    private final Runnable closer;

    private final CountDownLatch closedLatch = new CountDownLatch(1);
    private final CountDownLatch flushLatch = new CountDownLatch(1);

    public AsyncSampleSink(Consumer<SampleRow> sink, Runnable closer, int queueSize) {
        this.thread = new Thread("AsyncSampleSink") {
            @Override
            public void run() {
                process();
            }
        };
        thread.setDaemon(true);
        this.queue = new ArrayBlockingQueue<>(queueSize);
        this.sink = sink;
        this.closer = closer;
        thread.start();
    }

    @Override
    public void accept(SampleRow t) {
        if (closedLatch.getCount() == 0) {
            throw new IllegalArgumentException("Closed");
        }
        try {
            queue.put(t);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            return;
        }
    }

    public void close() {
        closedLatch.countDown();
        thread.interrupt();
    }

    public void join() throws InterruptedException {
        flushLatch.await();
    }

    private void process() {
        while(true) {
            try {
                SampleRow row = queue.poll(1, TimeUnit.SECONDS);
                if (row != null) {
                    feed(row);
                }
            } catch (InterruptedException e) {
            }
            if (closedLatch.getCount() == 0) {
                drainAndFlush();
                flushLatch.countDown();
                break;
            }
        }
    }

    private void feed(SampleRow row) {
        try {
            sink.accept(row);
        } catch (Exception e) {
            // ignore
        }
    }

    private void drainAndFlush() {
        while (true) {
            SampleRow row = queue.poll();
            if (row == null) {
                break;
            } else {
                feed(row);
            }
        }
        if (closer != null) {
            try {
                closer.run();
            } catch (Exception e) {
                // ignore
            }
        }
    }
}
