package org.gridkit.quickrun.report;

import java.time.Instant;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.LinkedHashMap;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import org.gridkit.quickrun.exec.TaskWatch;

public class ExecSample implements SampleRow, TaskWatch {

    private static final DateTimeFormatter TIMESTAMP = DateTimeFormatter.ofPattern("yyyy-MM-dd_HH:mm:ss.SSS").withLocale(Locale.ROOT).withZone(ZoneOffset.UTC);

    public static final String COL_TIMESTAMP = "TIMESTAMP";
    public static final String COL_STATUS = "STATUS";
    public static final String COL_DURATION = "DURATION";

    public static final String TASK_OK = "OK";
    public static final String TASK_ERROR = "ERROR";
    public static final String TASK_SKIP = "SKIP";

    private final String[] header;
    private final String[] values;
    private Instant timestamp;
    private long durationNS;
    private String status;

    public ExecSample() {
        this(new String[0], new String[0], Instant.now(), 0, "");
    }

    private ExecSample(String[] header, String[] values, Instant timestamp, long durationNS, String status) {
        this.header = header;
        this.values = values;
        this.timestamp = timestamp;
        this.durationNS = durationNS;
        this.status = status;
    }

    public ExecSample(Map<String, String> coords, Instant timestamp, long durationNS, String status) {
        this.header = coords.keySet().toArray(new String[0]);
        this.values = new String[header.length];
        for (int i = 0; i != header.length; ++i) {
            values[i] = coords.get(header[i]);
        }
        this.timestamp = timestamp;
        this.durationNS = durationNS;
        this.status = status;
    }

    @Override
    public ExecSample clone() {
        return new ExecSample(header, values, timestamp, durationNS, status);
    }

    @Override
    public void taskCompleted(Instant timestamp, long durationNS) {
        this.timestamp = timestamp;
        this.durationNS = durationNS;
        this.status = TASK_OK;
    }

    @Override
    public void taskFailed(Instant timestamp, long durationNS) {
        this.timestamp = timestamp;
        this.durationNS = durationNS;
        this.status = TASK_ERROR;
    }

    @Override
    public void taskSkipped(Instant timestamp) {
        this.timestamp = timestamp;
        this.durationNS = 0;
        this.status = TASK_SKIP;
    }

    public ExecSample with(String key, String val) {
        Map<String, String> map = new LinkedHashMap<>();
        for (int i = 0; i != header.length; ++i) {
            map.put(header[i], values[i]);
        }
        if (val == null) {
            map.remove(key);
        } else {
            map.put(key, val);
        }
        return new ExecSample(map, timestamp, durationNS, status);
    }

    @Override
    public void writeTo(KVSink sink) {
        for (int i = 0; i != header.length; ++i) {
            sink.append(header[i], values[i]);
        }
        String ts = TIMESTAMP.format(timestamp);
        sink.append(COL_TIMESTAMP, ts);
        String dur = durationNS == 0 ? "" : String.format("%.6f", ((double)durationNS) / TimeUnit.SECONDS.toNanos(1));
        sink.append(COL_DURATION, dur);
        sink.append(COL_STATUS, status);
    }
}
