package org.gridkit.quickrun.report;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

public class SimpleCSVSampleStore {

    private final File targetFile;
    private Header header = null;
    private Writer writer;

    public SimpleCSVSampleStore(File targetFile) {
        try {
            this.targetFile = targetFile;
            targetFile.delete();
            if (targetFile.exists()) {
                throw new IOException("Cannot remove the file");
            }
            if (targetFile.getParentFile() != null) {
                targetFile.getParentFile().mkdirs();
            }
            writer = openWriter();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private Writer openWriter() throws FileNotFoundException {
        return new OutputStreamWriter(new FileOutputStream(targetFile), StandardCharsets.UTF_8);
    }

    public synchronized void append(SampleRow row) {
        try {
            Map<String, String> data = new LinkedHashMap<String, String>();
            row.writeTo(data::put);
            if (header == null) {
                initHeader(data);
            } else {
                checkHeader(data);
            }
            appendRow(data);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public synchronized void close() {
        try {
            writer.flush();
            writer.close();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private void checkHeader(Map<String, String> data) throws IOException {
        boolean needRewrite = false;
        for (String key: data.keySet()) {
            if (!header.contains(key)) {
                needRewrite = true;
                break;
            }
        }

        if (!needRewrite) {
            return;
        }

        Header hdr = header.clone();
        for (String key: data.keySet()) {
            if (!hdr.contains(key)) {
                hdr.add(key);
            }
        }

        writer.close();
        File tmp = new File(targetFile.getParentFile(), targetFile.getName() + ".tmp");
        tmp.delete();
        if (tmp.exists()) {
            throw new IOException("Unable to create " + tmp.getPath());
        }
        targetFile.renameTo(tmp);

        targetFile.delete();
        writer = openWriter();

        header = hdr;
        writeHeader();

        rebuild(tmp);

        writer.flush();
        tmp.delete();
    }

    private void rebuild(File tmp) throws IOException {
        BufferedReader reader = new BufferedReader(new InputStreamReader(new FileInputStream(tmp), StandardCharsets.UTF_8));
        List<String> hdrLine = unescase(splitRow(reader.readLine()));
        Map<String, String> data = new HashMap<>();
        while(true) {
            String line = reader.readLine();
            if (line == null) {
                break;
            }
            data.clear();
            List<String> cells = splitRow(line);
            for (int n = 0; n < hdrLine.size(); ++n) {
                if (cells.size() > n) {
                    data.put(hdrLine.get(n), cells.get(n));
                }
            }
            appendRawRow(data);
        }
        reader.close();
    }

    private List<String> unescase(List<String> values) throws IOException {
        List<String> result = new ArrayList<>();
        for (String v: values) {
            result.add(unescape(v));
        }
        return result;
    }

    private String unescape(String val) throws IOException {
        if (val.indexOf('"') > 0) {
            val = val.trim();
            if (val.charAt(0) == '"' && val.charAt(val.length() - 1) == '"') {
                val = val.substring(1, val.length() - 1);
                val = val.replaceAll("[\\\\]n", "\n");
                val = val.replaceAll("[\\\\][\"]", "\n");
            } else {
                throw new IOException("Invalid value: " + val);
            }
        }
        return val;
    }

    private void initHeader(Map<String, String> data) throws IOException {
        Header hdr = new Header();
        for (String col: data.keySet()) {
            hdr.add(col);
        }

        header = hdr;

        writeHeader();
    }

    private void writeHeader() throws IOException {
        StringBuilder sb = new StringBuilder();
        for (String col: header) {
            formatCell(sb, col);
            sb.append(',');
        }
        sb.setLength(sb.length() - 1);
        sb.append("\n");
        writer.append(sb);
    }

    private void appendRow(Map<String, String> data) throws IOException {
        StringBuilder sb = new StringBuilder();
        for (String col: header) {
            formatCell(sb, data.get(col));
            sb.append(',');
        }
        sb.setLength(sb.length() - 1);
        sb.append("\n");
        writer.append(sb);
    }

    private void appendRawRow(Map<String, String> data) throws IOException {
        StringBuilder sb = new StringBuilder();
        for (String col: header) {
            String val = data.get(col);
            if (val != null) {
                sb.append(val);
            }
            sb.append(',');
        }
        sb.setLength(sb.length() - 1);
        sb.append("\n");
        writer.append(sb);
    }

    private void formatCell(Appendable out, String cell) {

        try {
            if (cell == null || cell.length() == 0) {
                return;
            }
            if (isCsvSafe(cell)) {
                out.append(cell);
            }
            else {
                out.append('"');
                for(int i = 0; i != cell.length(); ++i) {
                    char c = cell.charAt(i);
                    if (c == '"') {
                        out.append("\"\"");
                    }
                    else if (c == '\n') {
                        // replace line end with space
                        out.append(' ');
                    }
                    else {
                        out.append(c);
                    }
                }
                out.append('"');
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private List<String> splitRow(String line) {
        List<String> cells = new ArrayList<String>();
        int n = 0;
        while(n < line.length()) {
            int m = findEnd(line, n);
            cells.add(line.substring(n, m));
            n = m + 1;
        }
        return cells;
    }

    private int findEnd(String line, int n) {
        for (int i = n; i < line.length(); i++) {
            char ch = line.charAt(i);
            if (ch == ',') {
                return i;
            } else if (ch == '\"') {
                i = endOfQuote(line, i);
            }
        }
        return line.length();
    }

    private int endOfQuote(String line, int n) {
        for (int i = n + 1; i < line.length(); ++i) {
            char ch = line.charAt(i);
            if (ch == '\\') {
                ++i;
                continue;
            } else if (ch == '"') {
                return i;
            }

        }
        return line.length();
    }

    private boolean isCsvSafe(String row) {
        for(int i = 0; i != row.length(); ++i) {
            char ch = row.charAt(i);
            if (!Character.isJavaIdentifierPart(ch) && "._-".indexOf(ch) < 0) {
                return false;
            }
        }
        return true;
    }

    private static class Header implements Iterable<String> {

        private List<String> header = new ArrayList<>();
        private Map<String, Integer> index = new HashMap<>();

        @Override
        public Iterator<String> iterator() {
            return header.iterator();
        }

        @Override
        public Header clone() {
            Header hdr = new Header();
            hdr.header.addAll(header);
            hdr.index.putAll(index);
            return hdr;
        }

        public void add(String col) {
            if (!index.containsKey(col)) {
                header.add(col);
                index.put(col, index.size());
            }
        }

        public boolean contains(String col) {
            return index.containsKey(col);
        }
    }
}
