package org.gridkit.quickrun.report;

import java.time.Instant;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

import org.gridkit.quickrun.exec.TaskWatch;

public class SimpleTaskStats implements TaskWatch {

    private final ExecSample prototype;
    private final Consumer<SampleRow> sampleSink;

    private long totalTasks = 0;
    private long doneTasks = 0;
    private long errorTasks = 0;
    private long skipTasks = 0;

    private double doneTotalTime = 0;
    private double errorTotalTime = 0;

    public SimpleTaskStats(ExecSample prototype, Consumer<SampleRow> sampleSink) {
        this.prototype = prototype;
        this.sampleSink = sampleSink;
    }

    public synchronized long getTotalTasks() {
        return totalTasks;
    }

    public synchronized long getDoneTasks() {
        return doneTasks;
    }

    public synchronized long getErrorTasks() {
        return errorTasks;
    }

    public synchronized long getSkipTasks() {
        return skipTasks;
    }

    public synchronized double getDoneAvg() {
        return doneTotalTime / doneTasks;
    }

    public synchronized double getErrorAvg() {
        return errorTotalTime / errorTasks;
    }

    @Override
    public synchronized void taskCompleted(Instant timestamp, long durationNS) {
        totalTasks++;
        doneTasks++;
        doneTotalTime += ((double) durationNS) / TimeUnit.SECONDS.toNanos(1);

        if (sampleSink != null) {
            ExecSample s = prototype.clone();
            s.taskCompleted(timestamp, durationNS);
            sampleSink.accept(s);
        }
    }

    @Override
    public synchronized void taskFailed(Instant timestamp, long durationNS) {
        totalTasks++;
        errorTasks++;
        errorTotalTime += ((double) durationNS) / TimeUnit.SECONDS.toNanos(1);

        if (sampleSink != null) {
            ExecSample s = prototype.clone();
            s.taskFailed(timestamp, durationNS);
            sampleSink.accept(s);
        }
    }

    @Override
    public synchronized void taskSkipped(Instant timestamp) {
        totalTasks++;
        skipTasks++;

        if (sampleSink != null) {
            ExecSample s = prototype.clone();
            s.taskSkipped(timestamp);
            sampleSink.accept(s);
        }
    }
}
