package org.gridkit.quickrun.util;

import org.gridkit.nanoparser.NanoGrammar.SyntaticScope;
import org.gridkit.nanoparser.NanoParser;
import org.gridkit.nanoparser.ParserException;

import java.util.concurrent.TimeUnit;

import org.gridkit.nanoparser.NanoGrammar;
import org.gridkit.nanoparser.ReflectionActionSource;
import org.gridkit.nanoparser.Token;

public class TimeExpressionParser {

    private static final NanoParser<Long> TIME_EXPRESSION_PARSER = new NanoParser<>(TimeExpresseionHandeler.SYNTAX, new TimeExpresseionHandeler());

    public static long parseMillis(String expression) {
        Typed value = TIME_EXPRESSION_PARSER.parse(null, Typed.class, expression);
        return value.valueMs;
    }

    public static long parseAbsoluteMillis(long now, String expression) {
        Typed value = TIME_EXPRESSION_PARSER.parse(now, Typed.class, expression);
        return value.valueMs;
    }

    private static class TimeExpresseionHandeler extends ReflectionActionSource<Long> {

        public static SyntaticScope SYNTAX  = NanoGrammar.newParseTable()
            .skipSpace()
            .token("T")
            .term("NUMBER", "~[0-9]+(\\.[0-9]+)?")
            .postfixOp("TU", "~(ms|MS|s|S|m|M|h|H|d|D)").rank(2)
            .infixOp("ADD", "+").rank(1)
            .infixOp("SUB", "-").rank(1)
            .glueOp("ADD").rank(1)
            .toScope();

        @Term("T")
        public Typed now(@Source Token tkn, @Context Long now) {
            if (now == null) {
                throw new ParserException(tkn, "T is not defined");
            }
            return new Typed(now);
        }

        @Term("NUMBER")
        public Untyped number(String text) {
            return new Untyped(Double.parseDouble(text));
        }

        @Unary("TU")
        public Typed timeUnits(@Source Token op, Untyped arg) {
            String tu = op.tokenBody();
            switch (tu.toUpperCase()) {
                case "MS": return new Typed((long)arg.value);
                case "S":  return new Typed((long)(TimeUnit.SECONDS.toMillis(1) * arg.value));
                case "M":  return new Typed((long)(TimeUnit.MINUTES.toMillis(1) * arg.value));
                case "H":  return new Typed((long)(TimeUnit.HOURS.toMillis(1) * arg.value));
                case "D":  return new Typed((long)(TimeUnit.DAYS.toMillis(1) * arg.value));
                default: throw new ParserException(op, "Unknown type units");
            }
        }

        @Binary("ADD")
        public Typed add(Typed a, Typed b) {
            return new Typed(a.valueMs + b.valueMs);
        }

        @Binary("SUB")
        public Typed sub(Typed a, Typed b) {
            return new Typed(a.valueMs - b.valueMs);
        }
    }

    private static class Untyped {

        final double value;

        public Untyped(double value) {
            this.value = value;
        }

    }

    private static class Typed {

        final long valueMs;

        public Typed(long valueMs) {
            this.valueMs = valueMs;
        }
    }
}
