// $Id: ValidateTest.java 17111 2009-07-16 16:25:47Z hardy.ferentschik $
/*
* JBoss, Home of Professional Open Source
* Copyright 2008, Red Hat Middleware LLC, and individual contributors
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
* http://www.apache.org/licenses/LICENSE-2.0
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,  
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/
package org.hibernate.jsr303.tck.tests.validation;

import java.lang.annotation.Annotation;
import java.util.Set;
import javax.validation.ConstraintViolation;
import javax.validation.ValidationException;
import javax.validation.Validator;
import javax.validation.constraints.Pattern;
import javax.validation.groups.Default;
import javax.validation.metadata.BeanDescriptor;
import javax.validation.metadata.ConstraintDescriptor;
import javax.validation.metadata.PropertyDescriptor;

import org.jboss.test.audit.annotations.SpecAssertion;
import org.jboss.test.audit.annotations.SpecAssertions;
import org.jboss.testharness.AbstractTest;
import org.jboss.testharness.impl.packaging.Artifact;
import org.jboss.testharness.impl.packaging.ArtifactType;
import org.jboss.testharness.impl.packaging.Classes;
import static org.testng.Assert.assertEquals;
import static org.testng.Assert.assertNotNull;
import static org.testng.Assert.assertTrue;
import static org.testng.Assert.fail;
import org.testng.annotations.Test;

import org.hibernate.jsr303.tck.util.TestUtil;
import static org.hibernate.jsr303.tck.util.TestUtil.assertCorrectNumberOfViolations;
import static org.hibernate.jsr303.tck.util.TestUtil.assertCorrectPropertyPaths;

/**
 * Tests for the implementation of <code>Validator</code>.
 *
 * @author Hardy Ferentschik
 */
@Artifact(artifactType = ArtifactType.JSR303)
@Classes({ TestUtil.class, TestUtil.PathImpl.class, TestUtil.NodeImpl.class })
public class ValidateTest extends AbstractTest {

	@Test
	@SpecAssertions({
			@SpecAssertion(section = "3.1", id = "a"),
			@SpecAssertion(section = "3.1.2", id = "f"),
			@SpecAssertion(section = "5.1", id = "a")
	})
	public void testValidatedPropertyDoesNotFollowJavaBeansConvention() {
		try {
			Boy boy = new Boy();
			TestUtil.getDefaultValidator().validate( boy );
			fail();
		}
		catch ( ValidationException e ) {
			// success
		}

	}

	@SpecAssertion(section = "4.1.1", id = "b")
	@Test(expectedExceptions = IllegalArgumentException.class)
	public void testValidateWithNull() {
		Validator validator = TestUtil.getDefaultValidator();
		validator.validate( null );
	}

	@Test
	@SpecAssertions({
			@SpecAssertion(section = "4.1.1", id = "a"),
			@SpecAssertion(section = "4.1.1", id = "c")
	})

	public void testMultipleViolationOfTheSameType() {
		Validator validator = TestUtil.getDefaultValidator();

		Engine engine = new Engine();
		engine.setSerialNumber( "mail@foobar.com" );
		Set<ConstraintViolation<Engine>> constraintViolations = validator.validate( engine );
		assertEquals( constraintViolations.size(), 2, "Wrong number of constraints" );

		engine.setSerialNumber( "ABCDEFGH1234" );
		constraintViolations = validator.validate( engine );
		assertEquals( constraintViolations.size(), 1, "Wrong number of constraints" );

		engine.setSerialNumber( "ABCD-EFGH-1234" );
		constraintViolations = validator.validate( engine );
		assertEquals( constraintViolations.size(), 0, "Wrong number of constraints" );
	}

	@Test
	@SpecAssertion(section = "4.1.1", id = "c")
	public void testMultipleConstraintViolationOfDifferentTypes() {
		Validator validator = TestUtil.getDefaultValidator();

		Address address = new Address();
		address.setAddressline1( null );
		address.setAddressline2( null );
		address.setCity( "Llanfairpwllgwyngyllgogerychwyrndrobwyll-llantysiliogogogoch" ); //town in North Wales

		Set<ConstraintViolation<Address>> constraintViolations = validator.validate( address );
		assertEquals(
				constraintViolations.size(),
				3,
				"We should have been 2 @NotNull violations for addresslines and one @Length violation for city."
		);
	}

	@Test
	@SpecAssertions({
			@SpecAssertion(section = "3.1", id = "a"),
			@SpecAssertion(section = "4.2", id = "a"),
			@SpecAssertion(section = "4.2", id = "b"),
			@SpecAssertion(section = "4.2", id = "c"),
			@SpecAssertion(section = "4.2", id = "d"),
			@SpecAssertion(section = "4.2", id = "e"),
			@SpecAssertion(section = "4.2", id = "g")
	})
	public void testConstraintViolation() {
		Validator validator = TestUtil.getDefaultValidator();

		Engine engine = new Engine();
		engine.setSerialNumber( "mail@foobar.com" );
		engine.setSerialNumber( "ABCDEFGH1234" );
		Set<ConstraintViolation<Engine>> constraintViolations = validator.validate( engine );
		assertEquals( constraintViolations.size(), 1, "Wrong number of constraints" );

		ConstraintViolation<Engine> violation = constraintViolations.iterator().next();

		assertEquals( violation.getMessage(), "must match ^....-....-....$", "Wrong message" );
		assertEquals( violation.getMessageTemplate(), "must match {regexp}", "Wrong message template" );
		assertEquals( violation.getRootBean(), engine, "Wrong root entity." );
		assertEquals( violation.getInvalidValue(), "ABCDEFGH1234", "Wrong validated value" );
		assertNotNull( violation.getConstraintDescriptor(), "Constraint descriptor should not be null" );
		// cast is required for JDK 5 - at least on Mac OS X
		Annotation ann = ( Annotation ) violation.getConstraintDescriptor().getAnnotation();
		assertEquals( ann.annotationType(), Pattern.class, "Wrong annotation type" );
		assertCorrectPropertyPaths( constraintViolations, "serialNumber" );

		engine.setSerialNumber( "ABCD-EFGH-1234" );
		constraintViolations = validator.validate( engine );
		assertEquals( constraintViolations.size(), 0, "Wrong number of constraints" );
	}

	@Test
	@SpecAssertions({
			@SpecAssertion(section = "4.2", id = "f"),
			@SpecAssertion(section = "4.2", id = "g")
	})
	public void testValidateAssociation() {
		Validator validator = TestUtil.getDefaultValidator();

		Customer customer = new Customer();
		customer.setFirstName( "John" );
		customer.setLastName( "Doe" );
		Order order = new Order();
		customer.addOrder( order );

		Set<ConstraintViolation<Customer>> constraintViolations = validator.validate( customer );
		ConstraintViolation constraintViolation = constraintViolations.iterator().next();
		assertEquals( constraintViolations.size(), 1, "Wrong number of constraints" );
		assertEquals( constraintViolation.getRootBean(), customer, "Wrong root entity" );
		assertEquals( constraintViolation.getInvalidValue(), order.getOrderNumber(), "Wrong value" );
		assertCorrectPropertyPaths( constraintViolations, "orders[].orderNumber" );
	}

	@Test
	@SpecAssertions({
			@SpecAssertion(section = "2.4", id = "o"),
			@SpecAssertion(section = "4.2", id = "h")
	})
	public void testGraphValidationWithList() {
		Validator validator = TestUtil.getDefaultValidator();

		Actor clint = new ActorListBased( "Clint", "Eastwood" );
		Actor morgan = new ActorListBased( "Morgan", null );
		Actor charlie = new ActorListBased( "Charlie", "Sheen" );

		clint.addPlayedWith( charlie );
		charlie.addPlayedWith( clint );
		charlie.addPlayedWith( morgan );
		morgan.addPlayedWith( charlie );
		morgan.addPlayedWith( clint );
		clint.addPlayedWith( morgan );


		Set<ConstraintViolation<Actor>> constraintViolations = validator.validate( clint );
		assertEquals( constraintViolations.size(), 2, "Wrong number of constraints" );
		ConstraintViolation constraintViolation = constraintViolations.iterator().next();
		assertEquals( "Everyone has a last name.", constraintViolation.getMessage(), "Wrong message" );
		assertEquals( constraintViolation.getRootBean(), clint, "Wrong root entity" );
		assertEquals( constraintViolation.getInvalidValue(), morgan.getLastName(), "Wrong value" );
		assertCorrectPropertyPaths(
				constraintViolations, "playedWith[0].playedWith[1].lastName", "playedWith[1].lastName"
		);
	}

	@Test
	@SpecAssertions({
			@SpecAssertion(section = "2.4", id = "o"),
			@SpecAssertion(section = "3.1.3", id = "c"),
			@SpecAssertion(section = "4.2", id = "h")
	})
	public void testGraphValidationWithArray() {
		Validator validator = TestUtil.getDefaultValidator();

		Actor clint = new ActorArrayBased( "Clint", "Eastwood" );
		Actor morgan = new ActorArrayBased( "Morgan", null );
		Actor charlie = new ActorArrayBased( "Charlie", "Sheen" );

		clint.addPlayedWith( charlie );
		charlie.addPlayedWith( clint );
		charlie.addPlayedWith( morgan );
		morgan.addPlayedWith( charlie );
		morgan.addPlayedWith( clint );
		clint.addPlayedWith( morgan );

		Set<ConstraintViolation<Actor>> constraintViolations = validator.validate( clint );
		assertEquals( constraintViolations.size(), 2, "Wrong number of constraints" );
		ConstraintViolation constraintViolation = constraintViolations.iterator().next();
		assertEquals( "Everyone has a last name.", constraintViolation.getMessage(), "Wrong message" );
		assertEquals( constraintViolation.getRootBean(), clint, "Wrong root entity" );
		assertEquals( constraintViolation.getInvalidValue(), morgan.getLastName(), "Wrong value" );
		assertCorrectPropertyPaths(
				constraintViolations, "playedWith[0].playedWith[1].lastName", "playedWith[1].lastName"
		);
	}

	@Test(expectedExceptions = IllegalArgumentException.class)
	public void testNullParamterToValidatorImplConstructor() {
		TestUtil.getDefaultValidator().getConstraintsForClass( null );
	}

	@Test
	@SuppressWarnings("NullArgumentToVariableArgMethod")
	public void testPassingNullAsGroup() {
		Validator validator = TestUtil.getDefaultValidator();
		Customer customer = new Customer();
		try {
			validator.validate( customer, null );
		}
		catch ( IllegalArgumentException e ) {
			// success
		}
	}

	@Test
	public void testValidationIsPolymorphic() {
		Validator validator = TestUtil.getDefaultValidator();

		Customer customer = new Customer();
		customer.setFirstName( "Foo" );
		customer.setLastName( "Bar" );

		Order order = new Order();
		customer.addOrder( order );

		Set<ConstraintViolation<Person>> constraintViolations = validator.validate( ( Person ) customer );
		assertEquals( constraintViolations.size(), 1, "Wrong number of constraints" );

		TestUtil.assertConstraintViolation(
				constraintViolations.iterator().next(),
				Customer.class,
				null,
				"orders[].orderNumber"
		);

		order.setOrderNumber( 123 );

		constraintViolations = validator.validate( ( Person ) customer );
		assertEquals( constraintViolations.size(), 0, "Wrong number of constraints" );
	}

	@Test
	public void testObjectTraversion() {
		Validator validator = TestUtil.getDefaultValidator();

		Customer customer = new Customer();
		customer.setFirstName( "John" );
		customer.setLastName( "Doe" );

		for ( int i = 0; i < 100; i++ ) {
			Order order = new Order();
			customer.addOrder( order );
		}

		Set<ConstraintViolation<Customer>> constraintViolations = validator.validate(
				customer, Default.class, First.class, Second.class, Last.class
		);
		assertEquals( constraintViolations.size(), 100, "Wrong number of constraints" );
	}

	@Test
	public void testConstraintDescriptorWithoutExplicitGroup() {
		Validator validator = TestUtil.getDefaultValidator();

		BeanDescriptor beanDescriptor = validator.getConstraintsForClass( Order.class );
		PropertyDescriptor propertyDescriptor = beanDescriptor.getConstraintsForProperty( "orderNumber" );
		Set<ConstraintDescriptor<?>> descriptors = propertyDescriptor.getConstraintDescriptors();

		assertEquals( descriptors.size(), 1, "There should be only one constraint descriptor" );
		ConstraintDescriptor<?> descriptor = descriptors.iterator().next();
		Set<Class<?>> groups = descriptor.getGroups();
		assertTrue( groups.size() == 1, "There should be only one group" );
		assertEquals(
				groups.iterator().next(),
				Default.class,
				"The declared constraint does not explicitly define a group, hence Default is expected"
		);
	}

	@Test
	@SpecAssertion(section = "3.5", id = "b")
	public void testOnlyFirstGroupInSequenceGetEvaluated() {
		Validator validator = TestUtil.getDefaultValidator();
		Car car = new Car( "USd-298" );

		Set<ConstraintViolation<Car>> violations = validator.validateProperty(
				car, "licensePlateNumber", First.class, Second.class
		);
		assertCorrectNumberOfViolations( violations, 1 );

		car.setLicensePlateNumber( "USD-298" );
		violations = validator.validateProperty(
				car, "licensePlateNumber", First.class, Second.class
		);
		assertCorrectNumberOfViolations( violations, 0 );
	}

	class Car {
		@Pattern(regexp = "[A-Z][A-Z][A-Z]-[0-9][0-9][0-9]", groups = { First.class, Second.class })
		private String licensePlateNumber;

		Car(String licensePlateNumber) {
			this.licensePlateNumber = licensePlateNumber;
		}

		public String getLicensePlateNumber() {
			return licensePlateNumber;
		}

		public void setLicensePlateNumber(String licensePlateNumber) {
			this.licensePlateNumber = licensePlateNumber;
		}
	}

	interface First {
	}

	interface Second {
	}
}