// $Id: ValidateValueTest.java 16948 2009-06-25 10:10:13Z hardy.ferentschik $
/*
* JBoss, Home of Professional Open Source
* Copyright 2008, Red Hat Middleware LLC, and individual contributors
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
* http://www.apache.org/licenses/LICENSE-2.0
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,  
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/
package org.hibernate.jsr303.tck.tests.validation;

import java.util.Set;
import javax.validation.ConstraintViolation;
import javax.validation.Validator;

import org.jboss.testharness.AbstractTest;
import org.jboss.testharness.impl.packaging.Artifact;
import org.jboss.testharness.impl.packaging.ArtifactType;
import org.jboss.testharness.impl.packaging.Classes;
import static org.testng.Assert.assertEquals;
import static org.testng.Assert.fail;
import org.testng.annotations.Test;

import org.hibernate.jsr303.tck.util.TestUtil;
import static org.hibernate.jsr303.tck.util.TestUtil.assertCorrectPropertyPaths;

/**
 * Tests for the implementation of <code>Validator</code>.
 *
 * @author Hardy Ferentschik
 */
@Artifact(artifactType = ArtifactType.JSR303)
@Classes({TestUtil.class, TestUtil.PathImpl.class, TestUtil.NodeImpl.class})
public class ValidateValueTest extends AbstractTest {


	@Test
	@SuppressWarnings("NullArgumentToVariableArgMethod")
	public void testPassingNullAsGroup() {
		Validator validator = TestUtil.getDefaultValidator();
		Customer customer = new Customer();
		try {
			validator.validate( customer, null );
		}
		catch ( IllegalArgumentException e ) {
			// success
		}

		try {
			validator.validateProperty( customer, "firstName", null );
			fail();
		}
		catch ( IllegalArgumentException e ) {
			// success
		}

		try {
			validator.validateValue( Customer.class, "firstName", "foobar", null );
			fail();
		}
		catch ( IllegalArgumentException e ) {
			// success
		}
	}

	@Test
	public void testValidateWithNullProperty() {
		Validator validator = TestUtil.getDefaultValidator();
		try {
			validator.validate( null );
			fail();
		}
		catch ( IllegalArgumentException e ) {
			// success
		}

		try {
			validator.validateProperty( null, "firstName" );
			fail();
		}
		catch ( IllegalArgumentException e ) {
			// success
		}

		try {
			validator.validateValue( null, "firstName", "foobar" );
			fail();
		}
		catch ( IllegalArgumentException e ) {
			// success
		}
	}

	@Test
	public void testMultipleValidationMethods() {
		Validator validator = TestUtil.getDefaultValidator();

		Address address = new Address();
		address.setAddressline1( null );
		address.setAddressline2( null );
		address.setCity( "Llanfairpwllgwyngyllgogerychwyrndrobwyll-llantysiliogogogoch" ); //town in North Wales

		Set<ConstraintViolation<Address>> constraintViolations = validator.validate( address );
		assertEquals(
				constraintViolations.size(),
				3,
				"we should have been 2 not null violation for addresslines and one length violation for city"
		);

		constraintViolations = validator.validateProperty( address, "city" );
		assertEquals(
				constraintViolations.size(),
				1,
				"only city should be validated"
		);

		constraintViolations = validator.validateProperty( address, "city" );
		assertEquals(
				constraintViolations.size(),
				1,
				"only city should be validated"
		);

		constraintViolations = validator.validateValue( Address.class, "city", "Paris" );
		assertEquals(
				constraintViolations.size(),
				0,
				"Paris should be a valid city name."
		);
	}

	@Test
	public void testValidateValue() {
		Validator validator = TestUtil.getDefaultValidator();

		Set<ConstraintViolation<Customer>> constraintViolations = validator.validateValue(
				Customer.class, "orders[0].orderNumber", null
		);
		assertEquals( constraintViolations.size(), 1, "Wrong number of constraints" );

		ConstraintViolation constraintViolation = constraintViolations.iterator().next();
		assertEquals( constraintViolations.size(), 1, "Wrong number of constraints" );
		assertEquals( constraintViolation.getMessage(), "may not be null", "Wrong message" );
		assertEquals( constraintViolation.getRootBean(), null, "Wrong root entity" );
		assertEquals( constraintViolation.getRootBeanClass(), Customer.class, "Wrong root bean class" );
		assertEquals( constraintViolation.getInvalidValue(), null, "Wrong value" );
		assertCorrectPropertyPaths( constraintViolations, "orders[0].orderNumber" );

		constraintViolations = validator.validateValue( Customer.class, "orders[0].orderNumber", 1234 );
		assertEquals( constraintViolations.size(), 0, "Wrong number of constraints" );
	}

	@Test
	public void testValidateValueWithInvalidPropertyPath() {
		Validator validator = TestUtil.getDefaultValidator();

		try {
			validator.validateValue( Customer.class, "", null );
			fail();
		}
		catch ( IllegalArgumentException e ) {
			// success
		}

		try {
			validator.validateValue( Customer.class, "foobar", null );
			fail();
		}
		catch ( IllegalArgumentException e ) {
			// success
		}

		try {
			validator.validateValue( Customer.class, "orders[0].foobar", null );
			fail();
		}
		catch ( IllegalArgumentException e ) {
			// success
		}
	}
}