// $Id: XmlConfigurationTest.java 16956 2009-06-26 12:18:52Z hardy.ferentschik $
/*
* JBoss, Home of Professional Open Source
* Copyright 2008, Red Hat Middleware LLC, and individual contributors
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
* http://www.apache.org/licenses/LICENSE-2.0
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/
package org.hibernate.jsr303.tck.tests.xmlconfiguration;

import java.util.Set;
import javax.validation.ConstraintViolation;
import javax.validation.Validator;

import org.jboss.testharness.AbstractTest;
import org.jboss.testharness.impl.packaging.Artifact;
import org.jboss.testharness.impl.packaging.ArtifactType;
import org.jboss.testharness.impl.packaging.Classes;
import org.jboss.testharness.impl.packaging.Resource;
import org.jboss.testharness.impl.packaging.Resources;
import org.jboss.testharness.impl.packaging.jsr303.ValidationXml;
import org.testng.annotations.Test;

import org.hibernate.jsr303.tck.util.TestUtil;
import static org.hibernate.jsr303.tck.util.TestUtil.assertCorrectConstraintViolationMessages;
import static org.hibernate.jsr303.tck.util.TestUtil.assertCorrectNumberOfViolations;

/**
 * @author Hardy Ferentschik
 */
@Artifact(artifactType = ArtifactType.JSR303)
@Classes({ TestUtil.class, TestUtil.PathImpl.class, TestUtil.NodeImpl.class })
@ValidationXml(value = "validation.xml")
@Resources(
		{
				@Resource(source = "user-constraints.xml",
						destination = "WEB-INF/classes/org/hibernate/jsr303/tck/tests/xmlconfiguration/user-constraints.xml"),
				@Resource(source = "order-constraints.xml",
						destination = "WEB-INF/classes/org/hibernate/jsr303/tck/tests/xmlconfiguration/order-constraints.xml")
		}
)
public class XmlConfigurationTest extends AbstractTest {

	@Test
	public void testClassConstraintDefinedInXml() {
		Validator validator = TestUtil.getDefaultValidator();

		User user = new User();
		Set<ConstraintViolation<User>> constraintViolations = validator.validate( user );
		assertCorrectNumberOfViolations( constraintViolations, 1 );
		assertCorrectConstraintViolationMessages(
				constraintViolations, "Message from xml"
		);

		user.setConsistent( true );
		constraintViolations = validator.validate( user );
		assertCorrectNumberOfViolations( constraintViolations, 0 );
	}

	@Test
	public void testPropertyConstraintDefinedInXml() {
		Validator validator = TestUtil.getDefaultValidator();

		User user = new User();
		user.setConsistent( true );
		user.setFirstname( "Wolfeschlegelsteinhausenbergerdorff" );

		Set<ConstraintViolation<User>> constraintViolations = validator.validate( user );
		assertCorrectNumberOfViolations( constraintViolations, 1 );
		assertCorrectConstraintViolationMessages( constraintViolations, "Size is limited!" );

		user.setFirstname( "Wolfgang" );
		constraintViolations = validator.validate( user );
		assertCorrectNumberOfViolations( constraintViolations, 0 );
	}

	@Test
	public void testFieldConstraintDefinedInXml() {
		Validator validator = TestUtil.getDefaultValidator();

		User user = new User();
		user.setConsistent( true );
		user.setFirstname( "Wolfgang" );
		user.setLastname( "doe" );

		Set<ConstraintViolation<User>> constraintViolations = validator.validate( user );
		assertCorrectNumberOfViolations( constraintViolations, 1 );
		assertCorrectConstraintViolationMessages(
				constraintViolations, "Last name has to start with with a capital letter."
		);

		user.setLastname( "Doe" );
		constraintViolations = validator.validate( user );
		assertCorrectNumberOfViolations( constraintViolations, 0 );
	}

	@Test
	public void testAnnotationDefinedConstraintApplies() {
		Validator validator = TestUtil.getDefaultValidator();

		User user = new User();
		user.setConsistent( true );
		user.setPhoneNumber( "police" );

		Set<ConstraintViolation<User>> constraintViolations = validator.validate( user );
		assertCorrectNumberOfViolations( constraintViolations, 1 );
		assertCorrectConstraintViolationMessages(
				constraintViolations, "A phone number can only contain numbers, whitespaces and dashes."
		);

		user.setPhoneNumber( "112" );
		constraintViolations = validator.validate( user );
		assertCorrectNumberOfViolations( constraintViolations, 0 );
	}

	@Test
	public void testCascadingConfiguredInXml() {
		Validator validator = TestUtil.getDefaultValidator();

		User user = new User();
		user.setConsistent( true );
		CreditCard card = new CreditCard();
		card.setNumber( "not a number" );
		user.setCreditcard( card );

		Set<ConstraintViolation<User>> constraintViolations = validator.validate( user );
		assertCorrectNumberOfViolations( constraintViolations, 1 );
		assertCorrectConstraintViolationMessages( constraintViolations, "Not a credit casrd number." );

		card.setNumber( "1234567890" );
		constraintViolations = validator.validate( user );
		assertCorrectNumberOfViolations( constraintViolations, 0 );
	}

//	@Test(expectedExceptions = ValidationException.class)
//	public void testInvalidValidationXml() {
//		getValidatorWithCustomConfiguration( "META-INF/validation-invalid-xmlconfiguration.xmlconfiguration" );
//	}
//
//	@Test
//	public void testNoDefinedConstraints() {
//		Validator validator = getValidatorWithCustomConfiguration( "org/hibernate/validation/engine/xmlconfiguration/validation.xmlconfiguration" );
//		assertFalse(
//				validator.getConstraintsForClass( Order.class ).isBeanConstrained(), "Bean should be unsonstrained"
//		);
//	}
}
