package org.hnau.emitter

import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.runBlocking
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import org.hnau.base.data.Time
import org.hnau.base.data.box.Box
import org.hnau.base.data.box.unsafe.get
import org.hnau.base.data.box.unsafe.unsafe
import org.hnau.base.data.maybe.tryOrError
import org.hnau.base.extensions.*
import org.hnau.base.extensions.time.asSeconds
import org.hnau.base.extensions.time.now
import org.hnau.base.extensions.time.toLevelsString
import org.hnau.emitter.observing.ObservingEmitter


object Main {

    private var getCount = 0

    private suspend fun getValue(): String {
        delay(0.5.asSeconds)
        getCount++
        getCount.ifLessThan(2) {
            throw IllegalStateException("First request")
        }
        return "Result($getCount)"
    }

    private val asyncEmitter =
            object : ObservingEmitter<suspend () -> String>() {

                private val mutex = Mutex()

                private val cachedValue =
                        Box.unsafe<String>()

                private val getter: suspend () -> String = {
                    mutex.withLock { cachedValue.get { getValue() } }
                }

                override fun onAttached(observer: (suspend () -> String) -> Unit) {
                    super.onAttached(observer)
                    synchronized(cachedValue) {
                        cachedValue.checkValueExists(
                                ifExists = { observer { it } },
                                ifNotExists = { call(getter) }
                        )
                    }
                }


            }

    @JvmStatic
    fun main(args: Array<String>) = runBlocking<Unit> {

        val started = Time.now()

        observe(
                started = started,
                emitter = asyncEmitter,
                number = 1
        )

        delay(2.asSeconds)

        observe(
                started = started,
                emitter = asyncEmitter,
                number = 2
        )

        delay(2.asSeconds)

        observe(
                started = started,
                emitter = asyncEmitter,
                number = 3
        )

    }

    private fun log(
            started: Time,
            message: String
    ) {
        val text = "[${(Time.now() - started).toLevelsString()}] $message"
        println(text)
    }

    private fun observe(
            started: Time,
            number: Int,
            emitter: Emitter<suspend () -> String>
    ) {
        emitter.observe { lazySuspend ->
            Dispatchers.IO.launch {
                tryOrError { lazySuspend() }
                        .checkSuccess(
                                ifSuccess = {
                                    log(started, "[$number] Received result: $it")
                                },
                                ifError = {
                                    log(started, "[$number] Received error: $it")
                                }
                        )
            }
        }
    }

}