package org.hnau.emitter.extensions

import org.hnau.base.data.box.Box
import org.hnau.base.data.box.sync.sync
import org.hnau.base.extensions.boolean.ifTrue
import org.hnau.base.extensions.ifNotNull
import org.hnau.base.extensions.invoke
import org.hnau.base.extensions.it
import org.hnau.emitter.Emitter


inline fun <I, O> Emitter<I>.wrap(
        crossinline action: (value: I, observer: (O) -> Unit) -> Unit
) = Emitter.create<O> { observer ->
    observe { value -> action(value, observer) }
}

inline fun <I, O> Emitter<I>.map(
        crossinline converter: (I) -> O
) = wrap<I, O> { value, observer ->
    observer(converter(value))
}

inline fun <T> Emitter<T>.filter(
        crossinline predicate: (T) -> Boolean
) = wrap<T, T> { value, observer ->
    predicate(value).ifTrue { observer(value) }
}

inline fun <T> Emitter<T>.unique(
        crossinline comparator: (first: T, second: T) -> Boolean = { first, second -> first == second }
) = Emitter.create<T> { observer ->
    val oldValue = Box.sync<T>()
    observe { newValue ->
        oldValue.checkValueExists(
                ifExists = { !comparator(it, newValue) },
                ifNotExists = { true }
        ).ifTrue {
            oldValue.set(newValue)
            observer(newValue)
        }
    }
}

fun <T> Emitter<T>.mapToString() =
        map(Any?::toString)

inline fun <T, R> Emitter<T>.mapNotNull(
        crossinline converter: (T) -> R?
) = wrap<T, R> { value, observer ->
    converter(value).ifNotNull(observer)
}

fun <T> Emitter<T?>.filterNotNull() =
        mapNotNull(::it)

inline fun <T> Emitter<T>.callIf(
        crossinline predicate: (T) -> Boolean
) = wrap<T, Unit> { value, observer ->
    predicate(value).ifTrue { observer() }
}
