package org.hnau.emitter.observing

import org.hnau.base.extensions.boolean.ifFalse
import org.hnau.base.extensions.boolean.ifTrue
import org.hnau.emitter.Detacher
import org.hnau.emitter.Emitter
import java.util.concurrent.CopyOnWriteArraySet


abstract class ObservingEmitter<T> : Emitter<T> {

    private val observers = CopyOnWriteArraySet<(T) -> Unit>()

    override fun observe(
            observer: (T) -> Unit
    ): Detacher {
        synchronized(observers) {
            observers.ifEmpty(this::beforeFirstAttached)
            observers.add(observer)
        }
        onAttached(observer)
        return createDetacher(observer)
    }

    protected open fun call(value: T) = synchronized(observers) {
        observers.forEach { observer -> observer(value) }
    }

    protected open fun beforeFirstAttached() {}

    protected open fun onAttached(observer: (T) -> Unit) {}

    protected open fun onDetached(observer: (T) -> Unit) {}

    protected open fun afterLastDetached() {}

    private fun createDetacher(observer: (T) -> Unit) =
            Detacher.create { detach(observer) }

    private fun detach(
            observer: (T) -> Unit
    ) {
        val isEmpty = synchronized(observers) {
            observers.remove(observer).ifFalse { return }
            observers.isEmpty()
        }
        onDetached(observer)
        isEmpty.ifTrue(this::afterLastDetached)
    }

}