package org.hnau.emitter.observing.push.possible

import org.hnau.base.data.box.Box
import org.hnau.base.data.box.sync.sync
import org.hnau.emitter.Emitter
import org.hnau.emitter.utils.Detachers
import org.hnau.emitter.utils.observe


class CombineEmitter<A, B, R>(
        private val firstSource: Emitter<A>,
        private val secondSource: Emitter<B>,
        private val combinator: (A, B) -> R
) : PossibleEmitter<R>() {

    private val firstValue = Box.sync<A>()
    private val secondValue = Box.sync<B>()

    private val detachers = Detachers()

    @Suppress("UNCHECKED_CAST")
    private fun setIfNeed() = synchronized(this) {
        set(
                firstValue.checkValueExists(
                        ifExists = { first ->
                            secondValue.checkValueExists(
                                    ifExists = { second -> combinator(first, second) },
                                    ifNotExists = { return@synchronized }
                            )
                        },
                        ifNotExists = { return@synchronized }
                )
        )
    }

    override fun beforeFirstAttached() {
        super.beforeFirstAttached()
        firstSource.observe(detachers) { newA ->
            firstValue.set(newA)
            setIfNeed()
        }
        secondSource.observe(detachers) { newB ->
            secondValue.set(newB)
            setIfNeed()
        }
    }

    override fun afterLastDetached() {
        super.afterLastDetached()
        synchronized(this) {
            detachers.detach()
            firstValue.clear()
            secondValue.clear()
            clear()
        }

    }

}

fun <A, B, R> Emitter.Companion.combine(
        firstSource: Emitter<A>,
        secondSource: Emitter<B>,
        combinator: (A, B) -> R
): Emitter<R> = CombineEmitter(
        firstSource = firstSource,
        secondSource = secondSource,
        combinator = combinator
)

fun <T, O, R> Emitter<T>.combineWith(
        other: Emitter<O>,
        combinator: (T, O) -> R
) = Emitter.combine(
        firstSource = this,
        secondSource = other,
        combinator = combinator
)