package org.iworkz.genesis.vertx.common.context.impl;

import java.util.Deque;
import java.util.LinkedList;

import org.iworkz.genesis.vertx.common.context.CommandContext;
import org.iworkz.genesis.vertx.common.context.TransactionContext;
import org.iworkz.genesis.vertx.common.persistence.AbstractDao;

import io.vertx.core.AsyncResult;
import io.vertx.core.Future;

public class CommandContextImpl implements CommandContext {
	
	protected final Deque<TransactionContext> tcxStack = new LinkedList<>();
	
	private static TransactionContextImpl noopTcx = new TransactionContextImpl(null, null);

	@Override
	public TransactionContext getTransactionContext() {
		if (!tcxStack.isEmpty()) {
			return tcxStack.peek();
		} else {
			return null;
		}
	}

	@Override
	public <T> Future<T> beginTransactional(AbstractDao<?> dao, AsyncResult<T> res) {
		if (res.succeeded()) {
			return dao.beginTransaction(this)
					.map(tcx -> {
						tcxStack.push(tcx);
						return res.result();
					});
		} else {
			tcxStack.push(noopTcx);
			return Future.failedFuture(res.cause());
		}
	}
	
	@Override
	public <T> Future<T> endTransactional(AsyncResult<T> res) {
		TransactionContext tcx = tcxStack.pop();
		if (tcx == null) {
			return Future.failedFuture ("Failed to close because no transaction context available");
		}
		if (!isNoopTcx(tcx)) {
			Future<T> future;
			if (res.failed()) {
				future = tcx.rollback(res.cause());
			} else {
				future = tcx.commit(res.result());
			}
			return future.eventually(v -> tcx.close());
		} else {
			if (res.failed()) {
				return Future.failedFuture(res.cause());
			} else {
				return Future.succeededFuture(res.result());
			}
		}
	}
	
	protected boolean isNoopTcx(TransactionContext tcx) {
		return tcx == noopTcx;
	}

}
