package org.iworkz.genesis.vertx.common.context.impl;


import org.iworkz.genesis.vertx.common.context.CommandContext;
import org.iworkz.genesis.vertx.common.context.TransactionContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.AsyncResult;
import io.vertx.core.Future;
import io.vertx.sqlclient.SqlConnection;
import io.vertx.sqlclient.Transaction;

public class TransactionContextImpl extends CommandContextImpl implements TransactionContext {

    private static final Logger log = LoggerFactory.getLogger(TransactionContextImpl.class);

    private final CommandContext ccx;
    private final SqlConnection connection;

    private Transaction tx;

    public TransactionContextImpl(CommandContext ccx, SqlConnection connection) {
        this.ccx = ccx;
        this.connection = connection;
    }

    @Override
    public Transaction getTransaction() {
        return tx;
    }

    @Override
    public SqlConnection getConnection() {
        return connection;
    }

    @Override
    public Future<TransactionContext> begin() {
        return connection.begin().map(newTx -> {
            this.tx = newTx;
            return this;
        });
    }

    @Override
    public <T> Future<T> commit(T t) {
        if (tx != null) {
            return tx.commit().map(r -> {
                this.tx = null;
                return t;
            });
        } else {
            return Future.failedFuture("No transaction found to commit");
        }
    }

    @Override
    public <T> Future<T> rollback(Throwable cause) {
        if (tx != null) {
            return tx.rollback().compose(r -> {
                this.tx = null;
                return Future.failedFuture(cause);
            });
        } else {
            return Future.failedFuture("No transaction found to rollback");
        }
    }

    @Override
    public <T> Future<T> close(AsyncResult<T> res) {
        try {
            if (tx != null) {
                if (res.succeeded()) {
                    log.error("Can not close because transaction exists");
                    return Future.failedFuture("Can not close because transaction exists");
                } else {
                    return rollback(res.cause()).compose(v -> close()).map(res.result());
                }
            } else {
                return close().compose(v -> {
                    if (res.succeeded()) {
                        return Future.succeededFuture(res.result());
                    } else {
                        return Future.failedFuture(res.cause());
                    }
                });
            }
        } catch (Exception ex) {
            return Future.failedFuture(ex);
        }
    }

    @Override
    public <T> Future<T> close(T t) {
        try {
            if (tx != null) {
                return commit(t).compose(v -> close()).map(t);
            } else {
                return close().map(t);
            }
        } catch (Exception ex) {
            return Future.failedFuture(ex);
        }
    }

    protected CommandContext getCommandContext() {
        return ccx;
    }

	@Override
	public <T> Future<Void> close() {
		return connection.close();
	}

}
