package org.iworkz.genesis.vertx.common.controller;

import java.lang.reflect.Method;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.function.Function;

import javax.inject.Inject;

import org.iworkz.genesis.vertx.common.context.CommandContext;
import org.iworkz.genesis.vertx.common.context.EventBusRequestContext;
import org.iworkz.genesis.vertx.common.context.HttpRequestContext;
import org.iworkz.genesis.vertx.common.context.RESTContext;
import org.iworkz.genesis.vertx.common.exception.ForbiddenException;
import org.iworkz.genesis.vertx.common.exception.NotFoundException;
import org.iworkz.genesis.vertx.common.mapping.SimpleTypeMapper;
import org.iworkz.genesis.vertx.common.query.QuerySpecification;
import org.iworkz.genesis.vertx.common.stream.AsyncReadStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.databind.ObjectMapper;

import io.swagger.annotations.ApiOperation;
import io.vertx.core.AsyncResult;
import io.vertx.core.Future;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.eventbus.DeliveryOptions;
import io.vertx.core.eventbus.Message;
import io.vertx.core.json.JsonArray;
import io.vertx.core.json.JsonObject;
import io.vertx.ext.auth.User;
import io.vertx.ext.web.RoutingContext;
import io.vertx.ext.web.validation.RequestParameter;
import io.vertx.ext.web.validation.RequestParameters;

public abstract class AbstractController extends AbstractRESTController {

    private static final Logger log = LoggerFactory.getLogger(AbstractController.class);

    @Inject
    private ObjectMapper objectMapper;

    @Inject
    private SimpleTypeMapper simpleTypeMapper;

    @Override
    public void addControllerOperations() {
        for (Method method : getClass().getDeclaredMethods()) {
            ApiOperation apiOperation = method.getAnnotation(ApiOperation.class);
            if (apiOperation != null) {
                log.info("ApiOperation: " + apiOperation.value());
                if (Future.class.isAssignableFrom(method.getReturnType())) {
                    addControllerOperation(apiOperation.value(), createRequestProcessor(method), "Internal error");
                } else if (AsyncReadStream.class.isAssignableFrom(method.getReturnType())) {
                    addStreamingControllerOperation(apiOperation.value(), createStreamingRequestProcessor(method),
                            "Internal error");
                } else {
                    log.error("Api operation {} has unsupported return type: {}", method.getName(),
                            method.getReturnType().getCanonicalName());
                    throw new RuntimeException("Unsupported return type in API operation : " + method.getName());
                }
            }
        }
    }

    @SuppressWarnings("unchecked")
    protected <T> Function<RESTContext, Future<T>> createRequestProcessor(Method method) {
        return restCtx -> {
            try {
                return (Future<T>) method.invoke(this, restCtx);
            } catch (Exception e) {
                String errorMsg = "Failed to invoke API operation: " + method.getDeclaringClass().getCanonicalName() + "."
                        + method.getName();
                log.error(errorMsg, e);
                return Future.failedFuture(errorMsg);
//                if (e instanceof RuntimeException) {
//                    throw (RuntimeException) e;
//                } else {
//                }
            }
        };
    }

    @SuppressWarnings("unchecked")
    protected <T> Function<RESTContext, AsyncReadStream<T>> createStreamingRequestProcessor(Method method) {
        return restCtx -> {
            try {
                return (AsyncReadStream<T>) method.invoke(this, restCtx);
            } catch (Exception e) {
                String errorMsg = "Failed to invoke API operation: " + method.getName();
                log.error(errorMsg, e);
                throw new RuntimeException(errorMsg, e);
            }
        };
    }

    protected abstract CommandContext createCommandContext(RoutingContext ctx);

    protected String determineUserName(User user) {
        if (user != null) {
            JsonObject principal = user.principal();
            if (principal != null) {
                return principal.getString("username");
            }
        }
        return null;
    }

    protected UUID getUUIDParameter(RoutingContext ctx, String paramName) {
        String value = ctx.request().getParam(paramName);
        try {
            return value != null ? UUID.fromString(value) : null;
        } catch (Exception ex) {
            throw new RuntimeException("Can not read " + value + " as UUID", ex);
        }
    }

    protected boolean getBooleanParameter(RoutingContext ctx, String paramName) {
        String value = ctx.request().getParam(paramName);
        return value != null && "true".equalsIgnoreCase(value);
    }

    protected int getIntParameter(RoutingContext ctx, String paramName) {
        String value = ctx.request().getParam(paramName);
        return value != null ? Integer.parseInt(value) : 0;
    }

    protected long getLongParameter(RoutingContext ctx, String paramName) {
        String value = ctx.request().getParam(paramName);
        return value != null ? Long.parseLong(value) : 0L;
    }

    protected String getStringParameter(RoutingContext ctx, String paramName) {
        return ctx.request().getParam(paramName);
    }

    protected Future<Buffer> getBodyAsBuffer(RoutingContext ctx) {
        return ctx.request().resume().body();
    }

    protected <T> Future<T> getBodyObject(RoutingContext ctx, Class<T> objectType) {
        return getBodyAsBuffer(ctx).map(buffer -> decodeBuffer(buffer, objectType));
    }

    @Override
    protected <T> void handleHttpRequest(RoutingContext ctx, Function<RESTContext, Future<T>> requestProcessor,
                                         String defaultErrorMessage) {
        requestProcessor.apply(createRequestContext(ctx))
                .onComplete(res -> sendResponse(res, defaultErrorMessage, ctx));
    }

    @Override
    protected <T> void handleEventBusRequest(Message<Buffer> message, Function<RESTContext, Future<T>> requestProcessor,
                                             String defaultErrorMessage) {
        requestProcessor.apply(createRequestContext(message))
                .onComplete(res -> sendResponse(res, defaultErrorMessage, message));
    }

    @Override
    protected <T> void handleStreamingHttpRequest(RoutingContext ctx, Function<RESTContext, AsyncReadStream<T>> requestProcessor,
                                                  String defaultErrorMessage) {
        sendResponse(requestProcessor.apply(createRequestContext(ctx)), ctx);
        // .onComplete(res -> sendResponse(res, defaultErrorMessage, ctx));
    }

    @Override
    protected <T> void handleStreamingEventBusRequest(Message<Buffer> message,
                                                      Function<RESTContext, AsyncReadStream<T>> requestProcessor,
                                                      String defaultErrorMessage) {
        requestProcessor.apply(createRequestContext(message)).toList() // TODO find a solution to stream result over EventBus
                .onComplete(res -> sendResponse(res, defaultErrorMessage, message));
    }

    protected RESTContext createRequestContext(RoutingContext ctx) {
        return new HttpRequestContext(ctx.request(), ctx.user(), ctx.data(), simpleTypeMapper);
    }

    protected RESTContext createRequestContext(Message<Buffer> message) {
        return new EventBusRequestContext(message, getUser(message), getMetaData(message), simpleTypeMapper);
    }

    protected User getUser(Message<Buffer> message) {
        return null;
    }

    protected Map<String, Object> getMetaData(Message<Buffer> message) {
        return Collections.emptyMap();
    }

    protected <T> void sendResponse(AsyncResult<T> res, String errorMessage, Message<Buffer> message) {
        if (res.succeeded()) {
            try {
                T t = res.result();
                if (t == null) {
                    message.reply(null);
                } else if (t instanceof List) {
                    DeliveryOptions options = setReplyHeader(new DeliveryOptions(), "content-type", "application/json");
                    message.reply(new JsonArray((List<?>) t).encodePrettily(), options);
                } else if (t instanceof JsonObject) {
                    DeliveryOptions options = setReplyHeader(new DeliveryOptions(), "content-type", "application/json");
                    message.reply(((JsonObject) t).encodePrettily(), options);
                } else if (t instanceof Boolean) {
                    message.reply(((Boolean) t).toString());
                } else if (t instanceof UUID) {
                    message.reply(((UUID) t).toString());
                } else {
                    DeliveryOptions options = setReplyHeader(new DeliveryOptions(), "content-type", "application/json");
                    String json = encodeObject(t);
                    message.reply(json, options);
                }
            } catch (Exception ex) {
                DeliveryOptions options = setReplyHeader(new DeliveryOptions(), "errorMessage", "application/json");
                message.reply(errorMessage, options);
            }
        } else {
            DeliveryOptions options = setReplyHeader(new DeliveryOptions(), "errorMessage", "application/json");
            message.reply(errorMessage, options);
        }
    }


    protected DeliveryOptions setReplyHeader(DeliveryOptions deliveryOptions, String name, String value) {
        deliveryOptions.addHeader(name, value);
        return deliveryOptions;
    }


    protected QuerySpecification buildQuerySpecification(RESTContext requestContext, String... filterCriteriaNames) {
        QuerySpecification querySpec = new QuerySpecification();

        Integer page = requestContext.getIntegerParam("page");
        if (page != null) {
            querySpec.setPage(page);
        }
        Integer pageSizeParameter = requestContext.getIntegerParam("pageSize");
        if (pageSizeParameter != null) {
            querySpec.setPageSize(pageSizeParameter);
        }
        String sortDirParameter = requestContext.getStringParam("sortDir");
        if (sortDirParameter != null) {
            querySpec.setSortBy(sortDirParameter);
        }

        Boolean sortByParameter = requestContext.getBooleanParam("descending");
        if (sortByParameter != null) {
            querySpec.setDescending(sortByParameter);
        }

        // querySpec.setPage(getIntParameter(ctx, "page"));
        // querySpec.setPageSize(getIntParameter(ctx, "pageSize"));
        // querySpec.setFilter(getStringParameter(ctx, "filter"));
        // querySpec.setSortBy(getStringParameter(ctx, "sortBy"));
        // querySpec.setDescending(getBooleanParameter(ctx, "descending"));
        for (String name : filterCriteriaNames) {
            Object requestParameter = requestContext.getParam(name);
            if (requestParameter != null) {
                querySpec.addFilterCriteria(name, requestParameter);
            }
        }
        return querySpec;
    }

    protected QuerySpecification buildQuerySpecification(RequestParameters parameters, String... filterCriteriaNames) {
        QuerySpecification querySpec = new QuerySpecification();

        RequestParameter pageParameter = parameters.queryParameter("page");
        if (pageParameter != null) {
            querySpec.setPage(pageParameter.getInteger());
        }
        RequestParameter pageSizeParameter = parameters.queryParameter("pageSize");
        if (pageSizeParameter != null) {
            querySpec.setPageSize(pageSizeParameter.getInteger());
        }
        RequestParameter sortDirParameter = parameters.queryParameter("sortDir");
        if (sortDirParameter != null) {
            querySpec.setSortBy(sortDirParameter.getString());
        }

        RequestParameter sortByParameter = parameters.queryParameter("descending");
        if (sortByParameter != null) {
            querySpec.setDescending(sortByParameter.getBoolean());
        }

        // querySpec.setPage(getIntParameter(ctx, "page"));
        // querySpec.setPageSize(getIntParameter(ctx, "pageSize"));
        // querySpec.setFilter(getStringParameter(ctx, "filter"));
        // querySpec.setSortBy(getStringParameter(ctx, "sortBy"));
        // querySpec.setDescending(getBooleanParameter(ctx, "descending"));
        for (String name : filterCriteriaNames) {
            RequestParameter requestParameter = parameters.queryParameter(name);
            if (requestParameter != null) {
                querySpec.addFilterCriteria(name, requestParameter.get());
            }
        }
        return querySpec;
    }

    protected QuerySpecification buildQuerySpecification(RoutingContext ctx) {
        QuerySpecification querySpec = new QuerySpecification();
        querySpec.setPage(getIntParameter(ctx, "page"));
        querySpec.setPageSize(getIntParameter(ctx, "pageSize"));
        querySpec.setFilter(getStringParameter(ctx, "filter"));
        querySpec.setSortBy(getStringParameter(ctx, "sortBy"));
        querySpec.setDescending(getBooleanParameter(ctx, "descending"));
        return querySpec;
    }


    protected <T> Future<Void> sendResponse(AsyncReadStream<T> readStream, RoutingContext ctx) {
        return sendResponse(readStream, this::encodeObject, ctx);
    }

    protected <T> Future<Void> sendResponse(AsyncReadStream<T> readStream, Function<T, String> mapper, RoutingContext ctx) {

        readStream.exceptionHandler(ex -> {
            sendErrorResponse(ctx, ex);
        });

        ctx.response().putHeader("content-type", "application/json");
        JsonArrayStreamResponse<T> streamResponse = new JsonArrayStreamResponse<>(ctx.response(), mapper);
        return readStream.pipeTo(streamResponse, false)
                .onFailure(ex -> {
                    sendErrorResponse(ctx, ex);
                });
    }

    protected <T> T decodeBuffer(Buffer buffer, Class<T> objectType) {
        try {
            if (buffer != null && buffer.length() > 0) {
                return getObjectMapper().readValue(buffer.toString(), objectType);
            } else {
                return null;
            }
        } catch (Exception e) {
            throw new RuntimeException("Failed to decode buffer to object of type: " + objectType.getCanonicalName(), e);
        }
    }

    protected <T> String encodeObject(T t) {
        try {
            if (t != null) {
                return getObjectMapper().writeValueAsString(t);
            } else {
                return null;
            }
        } catch (Exception e) {
            throw new RuntimeException("Failed to map buffer to object", e);
        }
    }

    protected <T> void sendResponse(AsyncResult<T> t, RoutingContext ctx) {
        sendResponse(t, null, ctx);
    }

    protected <T> void sendResponse(AsyncResult<T> res, String errorMessage, RoutingContext ctx) {
        if (res.succeeded()) {
            try {

                T t = res.result();
                if (t == null) {
                    ctx.response().end();
                } else if (t instanceof List) {
                    ctx.response().putHeader("content-type", "application/json");
                    ctx.response().end(new JsonArray((List<?>) t).encodePrettily());
                } else if (t instanceof JsonObject) {
                    ctx.response().putHeader("content-type", "application/json");
                    ctx.response().end(((JsonObject) t).encodePrettily());
                } else if (t instanceof Boolean) {
                    ctx.response().end(((Boolean) t).toString());
                } else if (t instanceof UUID) {
                    ctx.response().end(((UUID) t).toString());
                } else {
                    ctx.response().putHeader("content-type", "application/json");
                    String json = encodeObject(t);
                    ctx.response().end(json);
                }
            } catch (Exception ex) {
                sendErrorResponse(ctx, ex, errorMessage);
            }
        } else {
            sendErrorResponse(ctx, res.cause(), errorMessage);
        }
    }

    protected void sendErrorResponse(RoutingContext ctx, Throwable cause) {
        sendErrorResponse(ctx, cause, null);
    }

    protected void sendErrorResponse(RoutingContext ctx, Throwable cause, String errorMessage) {
        try {

            if (cause instanceof NotFoundException) {
                if (errorMessage == null) {
                    errorMessage = "Not found";
                }
                log.error(errorMessage, cause);
                ctx.response().setStatusCode(404).end(errorMessage);
            } else if (cause instanceof ForbiddenException) {
                if (errorMessage == null) {
                    errorMessage = "Forbidden";
                }
                log.error(errorMessage, cause);
                ctx.response().setStatusCode(403).end(errorMessage);
            } else {
                if (errorMessage == null) {
                    errorMessage = "Internal server error";
                }
                log.error(errorMessage, cause);
                ctx.response().setStatusCode(500).end(errorMessage);
            }
        } catch (Exception ex) {
            log.error("Failed to send error response", ex);
            ctx.response().setStatusCode(500).end("Internal server error");
        }
    }

    protected ObjectMapper getObjectMapper() {
        return objectMapper;
    }

    protected void handleNotImplemented(RoutingContext ctx) {
        getBodyAsBuffer(ctx)
                .map(buffer -> {
                    log.info("{}", buffer);
                    ctx.response().setChunked(true);
                    ctx.response().setStatusCode(501);
                    return buffer;
                })
                .onFailure(cause -> {
                    log.error("Internal server error", cause);
                    ctx.response().setStatusCode(500);
                })
                .onComplete(res -> ctx.response().end());
    }

}
