package org.iworkz.genesis.vertx.common.controller;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;

import javax.inject.Inject;

import org.iworkz.genesis.vertx.common.context.RESTContext;
import org.iworkz.genesis.vertx.common.router.RESTOperation;
import org.iworkz.genesis.vertx.common.stream.AsyncReadStream;

import io.vertx.core.Future;
import io.vertx.core.Handler;
import io.vertx.core.Vertx;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.eventbus.Message;
import io.vertx.ext.auth.User;
import io.vertx.ext.web.RoutingContext;

public abstract class AbstractRESTController implements RESTController {

    @Inject
    private Vertx vertx;

    private List<RESTOperation> controllerOperations;

    @Override
    public List<RESTOperation> getRESTOperations() {
        if (controllerOperations == null) {
            controllerOperations = new ArrayList<>();
            addControllerOperations();
        }
        return controllerOperations;
    }

    protected Vertx getVertx() {
        return vertx;
    }

    protected <T> void addControllerOperation(String operationId, Function<RESTContext, Future<T>> requestProcessor,
                                              String defaultErrorMessage) {
        RESTOperation controllerOperation = createControllerOperation(
                operationId,
                ctx -> handleHttpRequest(ctx, requestProcessor, defaultErrorMessage),
                message -> handleEventBusRequest(message, requestProcessor, defaultErrorMessage));
        controllerOperations.add(controllerOperation);
    }

    protected <T> void addStreamingControllerOperation(String operationId,
                                                       Function<RESTContext, AsyncReadStream<T>> requestProcessor,
                                                       String defaultErrorMessage) {
        RESTOperation controllerOperation = createControllerOperation(
                operationId,
                ctx -> handleStreamingHttpRequest(ctx, requestProcessor, defaultErrorMessage),
                message -> handleStreamingEventBusRequest(message, requestProcessor, defaultErrorMessage));
        controllerOperations.add(controllerOperation);
    }

    protected RESTOperation createControllerOperation(String operationId, Handler<RoutingContext> httpHandler,
                                                      Handler<Message<Buffer>> eventBusHandler) {
        return new RESTOperation(operationId, httpHandler, eventBusHandler);
    }

    protected abstract <T> void handleHttpRequest(RoutingContext ctx, Function<RESTContext, Future<T>> requestProcessor,
                                                  String defaultErrorMessage);

    protected abstract <T> void handleEventBusRequest(Message<Buffer> message,
                                                      Function<RESTContext, Future<T>> requestProcessor,
                                                      String defaultErrorMessage);

    protected abstract <T> void handleStreamingHttpRequest(RoutingContext ctx,
                                                           Function<RESTContext, AsyncReadStream<T>> requestProcessor,
                                                           String defaultErrorMessage);

    protected abstract <T> void handleStreamingEventBusRequest(Message<Buffer> message,
                                                               Function<RESTContext, AsyncReadStream<T>> requestProcessor,
                                                               String defaultErrorMessage);

    protected abstract void addControllerOperations();

}
