package org.iworkz.genesis.vertx.common.controller;

import java.util.function.Function;

import io.vertx.core.AsyncResult;
import io.vertx.core.Future;
import io.vertx.core.Handler;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.http.HttpServerResponse;
import io.vertx.core.streams.WriteStream;

public class JsonArrayStreamResponse<T> implements WriteStream<T> {

    private static final int INITIAL_MAX_JUNK_OBJECT_COUNT = 10;
    private static final int DEFAULT_MAX_JUNK_OBJECT_COUNT = 100;

    private final HttpServerResponse response;
    private final Function<T, String> mapper;

    private boolean isFirstElement = true;

    private final int initialMaxJunkObjectCount;
    private final int defaultMaxJunkObjectCount;
    private int junkNumber = 0;
    private int maxJunkObjectCount;
    private int junkObjectCount;

    private Buffer junk = Buffer.buffer();

    public JsonArrayStreamResponse(HttpServerResponse response, Function<T, String> mapper) {
        this(response, mapper, -1, -1);
    }

    public JsonArrayStreamResponse(HttpServerResponse response,
                                   Function<T, String> mapper,
                                   int initialMaxJunkObjectCount,
                                   int defaultMaxJunkObjectCount) {
        this.response = response;
        this.response.setChunked(true);
        this.mapper = mapper;
        if (initialMaxJunkObjectCount > 0) {
            this.initialMaxJunkObjectCount = initialMaxJunkObjectCount;
        } else {
            this.initialMaxJunkObjectCount = INITIAL_MAX_JUNK_OBJECT_COUNT;
        }
        if (defaultMaxJunkObjectCount > 0) {
            this.defaultMaxJunkObjectCount = defaultMaxJunkObjectCount;
        } else {
            this.defaultMaxJunkObjectCount = DEFAULT_MAX_JUNK_OBJECT_COUNT;
        }
        maxJunkObjectCount = getMaxJunkObjectCount(this.junkNumber);
    }

    @Override
    public WriteStream<T> exceptionHandler(Handler<Throwable> handler) {
        response.exceptionHandler(handler);
        return this;
    }

    @Override
    public Future<Void> write(T data) {
        addToJunk(data);
        if (junkObjectCount == maxJunkObjectCount) {
            return response.write(takeCurrentJunk());
        } else {
            return Future.succeededFuture();
        }
    }

    @Override
    public void write(T data, Handler<AsyncResult<Void>> handler) {
        addToJunk(data);
        if (junkObjectCount == maxJunkObjectCount) {
            response.write(takeCurrentJunk(), handler);
        }
    }

    protected Buffer takeCurrentJunk() {
        junkNumber++;
        Buffer currentJunk = junk;
        junkObjectCount = 0;
        maxJunkObjectCount = getMaxJunkObjectCount(junkNumber);
        junk = Buffer.buffer();
        return currentJunk;
    }

    protected int getMaxJunkObjectCount(int currentJunkNumber) {
        if (currentJunkNumber == 1) {
            return initialMaxJunkObjectCount;
        } else {
            return defaultMaxJunkObjectCount;
        }
    }

    protected void addToJunk(T data) {
        junkObjectCount++;
        if (isFirstElement) {
            isFirstElement = false;
            junk.appendString(startArray() + "\n  " + mapper.apply(data));
        } else {
            junk.appendString(",\n  " + mapper.apply(data));
        }
    }

    protected String startArray() {
        return "[";
    }

    protected String endArray() {
        return "]";
    }

    @Override
    public void end(Handler<AsyncResult<Void>> handler) {
        if (isFirstElement) {
            response.write(startArray() + endArray());
        } else {
            response.write(takeCurrentJunk() + "\n" + endArray());
        }
        if (handler != null) {
            response.end(handler);
        } else {
            response.end();
        }
    }

    @Override
    public WriteStream<T> setWriteQueueMaxSize(int maxSize) {
        response.setWriteQueueMaxSize(maxSize);
        return this;
    }

    @Override
    public boolean writeQueueFull() {
        return response.writeQueueFull();
    }

    @Override
    public WriteStream<T> drainHandler(Handler<Void> handler) {
        response.drainHandler(handler);
        return this;
    }

}
