package org.iworkz.genesis.vertx.common.factory;

import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.function.Supplier;

import org.iworkz.common.exception.GenesisException;


public abstract class AbstractFactory {

    protected final Set<Class<?>> suppliedClasses = new HashSet<>();
    private final Map<Class<?>, Supplier<?>> suppliers = new HashMap<>();


    public Set<Class<?>> getSuppliedClasses() {
        return suppliedClasses;
    }

    public <T> Supplier<T> getSupplier(Class<T> clazz) {
        try {
            Supplier<T> supplier = getExistingSupplier(clazz);
            if (supplier == null) {
                supplier = getOrCreateSupplierSynchronized(clazz);
            }
            return supplier;
        } catch (Exception ex) {
            throw new RuntimeException("Failed to get supplier for class '" + clazz.getCanonicalName() + "'", ex);
        }
    }

    protected <T> Supplier<T> getOrCreateSupplierSynchronized(Class<T> clazz) {
        synchronized (suppliers) {
            Supplier<T> supplier = getExistingSupplier(clazz);
            if (supplier == null) {
                supplier = createSupplierFor(clazz);
                if (supplier == null) {
                    throw new IllegalStateException(
                            "Creating a supplier not implemented for class '" + clazz.getCanonicalName() + "'");
                }
                suppliers.put(clazz, supplier);
                return supplier;
            }
            return supplier;
        }
    }

    @SuppressWarnings("unchecked")
    protected <T> Supplier<T> getExistingSupplier(Class<T> clazz) {
        return (Supplier<T>) suppliers.get(clazz);
    }

    @SuppressWarnings("unchecked")
    protected <T> Supplier<T> createSupplierFor(Class<T> clazz) {
        String createMethodName = "create" + clazz.getSimpleName();
        for (Method method : getClass().getDeclaredMethods()) {
            if (isCreateMethod(method, createMethodName, clazz)) {
                return () -> {
                    try {
                        return (T) method.invoke(this);
                    } catch (Exception ex) {
                        throw new GenesisException("Failed to invoke create method '" + createMethodName + "'", ex);
                    }
                };
            }
        }
        return null;
    }

    protected boolean isCreateMethod(Method method, String createMethodName, Class<?> clazz) {
        if (method.getParameterCount() == 0) {
            Class<?> returnType = method.getReturnType();
            boolean isAssignable = returnType != null && returnType.isAssignableFrom(clazz);
            return isAssignable && createMethodName.equals(method.getName());
        }
        return false;
    }

    protected <T> Supplier<T> supplierForCreateMethod(Supplier<T> createMethodReference) {
        return createMethodReference;
    }

}
