package org.iworkz.genesis.vertx.common.persistence;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;

import javax.inject.Inject;

import org.iworkz.genesis.vertx.common.context.CommandContext;
import org.iworkz.genesis.vertx.common.context.TransactionContext;
import org.iworkz.genesis.vertx.common.context.impl.TransactionContextImpl;
import org.iworkz.genesis.vertx.common.exception.NotFoundException;
import org.iworkz.genesis.vertx.common.factory.AbstractFactory;
import org.iworkz.genesis.vertx.common.query.QuerySpecification;
import org.iworkz.genesis.vertx.common.stream.AsyncReadStream;
import org.iworkz.genesis.vertx.common.stream.IterableReadStream;
import org.iworkz.genesis.vertx.common.stream.MappedRowReadStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.CompositeFuture;
import io.vertx.core.Future;
import io.vertx.core.Promise;
import io.vertx.core.Vertx;
import io.vertx.core.buffer.Buffer;
import io.vertx.sqlclient.Pool;
import io.vertx.sqlclient.PreparedQuery;
import io.vertx.sqlclient.Row;
import io.vertx.sqlclient.RowSet;
import io.vertx.sqlclient.SqlClient;
import io.vertx.sqlclient.SqlConnection;
import io.vertx.sqlclient.Tuple;
import io.vertx.sqlclient.impl.ArrayTuple;


public abstract class AbstractDao<T extends GenesisEntity> implements GenesisDao<T> {

    private static final Logger log = LoggerFactory.getLogger(AbstractDao.class);

    private static final int READ_ONCE_LIMIT = 50;

    private final Class<T> entityClass;

    protected AbstractDao(Class<T> entityClass) {
        this.entityClass = entityClass;
    }

    @Inject
    private Vertx vertx;

    @Inject
    private Pool client;

    public Future<Buffer> getDefaultSchemaContribution(String dbms, SchemaContributionType type) {
        String defaultSchemaContributionId = getDefaultSchemaContributionId(dbms);
        String fileName = getSchemaContributionFileName(dbms, type, defaultSchemaContributionId);
        if (fileName != null) {
            return vertx.fileSystem().readFile(fileName);
        } else {
            return Future.succeededFuture(Buffer.buffer());
        }
    }

    public Future<Buffer> getSchemaContribution(String dbms, SchemaContributionType type, String schemaContributionId) {
        String fileName = getSchemaContributionFileName(dbms, type, schemaContributionId);
        if (fileName != null) {
            return vertx.fileSystem().readFile(fileName);
        } else {
            return Future.succeededFuture(Buffer.buffer());
        }
    }

    protected String getDefaultSchemaContributionId(String dbms) {
        String[] schemaContributionIds = getSchemaContributionIds(dbms);
        if (schemaContributionIds != null && schemaContributionIds.length >= 1) {
            return schemaContributionIds[0];
        } else {
            return null;
        }
    }

    protected String getSchemaContributionFileName(String dbms, SchemaContributionType type, String schemaContributionId) {
        if (schemaContributionId != null) {
            String commonPartOfFileName = "database/" + schemaContributionId;
            switch (type) {
                case CREATE:
                    return commonPartOfFileName + "-create.sql";
                case DROP:
                    return commonPartOfFileName + "-drop.sql";
                default:
                    throw new RuntimeException("Type of schema contribution is not supported: " + type);
            }
        } else {
            return null;
        }
    }

    public String[] getSchemaContributionIds(String dbms) {
        return null;  // default is no schema contribution
    }

    public Future<TransactionContext> beginTransaction(CommandContext ccx) {
        return getPool(ccx)
                .compose(Pool::getConnection)
                .compose(connection -> {
                    TransactionContext tcx = createTransactionContext(ccx, connection);
                    return tcx.begin();
                });
    }

    protected <T> T generateId(Class<T> keyClass) {
        if (UUID.class == keyClass) {
            return (T) UUID.randomUUID();
        } else if (String.class == keyClass) {
            return (T) UUID.randomUUID().toString();
        } else {
            throw new RuntimeException("Generation of class " + keyClass.getCanonicalName() + " not implemented");
        }
    }

    protected TransactionContext createTransactionContext(CommandContext ccx, SqlConnection connection) {
        return new TransactionContextImpl(ccx, connection);
    }

    public Future<Pool> getPool() {
        return getPool(null);
    }

    public Future<Pool> getPool(CommandContext ctx) {
        return Future.succeededFuture(client);
    }

    public Future<SqlClient> getClient() {
        return getClient(null);
    }

    public Future<SqlClient> getClient(CommandContext ctx) {
        if (ctx != null) {
            TransactionContext tcx = (ctx instanceof TransactionContext) ? (TransactionContext) ctx : ctx.getTransactionContext();
            if (tcx != null) {
                return Future.succeededFuture(tcx.getConnection());
            }
        }
        return getPool(ctx).map(pool -> pool);
    }
    //
    // public MeterRegistry getRegistry() {
    // return registry;
    // }

    public Class<T> getEntityClass() {
        return entityClass;
    }

    public AbstractFactory getFactory() {
        return null;
    }

    public String getEntityPackageName() {
        final Class<T> entityClassOrNull = getEntityClass();
        if (entityClassOrNull != null) {
            return entityClassOrNull.getPackage().getName();
        } else {
            String daoPackageName = getClass().getPackage().getName();
            return daoPackageName.replace(".dao", ".entities");
        }
    }

    protected Future<RowSet<Row>> executePreparedQuery(CommandContext ccx, String sqlCommand) {
        try {
            return getClient(ccx).compose(c -> c.preparedQuery(sqlCommand).execute());
        } catch (Exception ex) {
            return Future.failedFuture(ex);
        }
    }

    protected Future<RowSet<Row>> executePreparedQuery(CommandContext ccx, String sqlCommand, Tuple parameters) {
        try {
            return getClient(ccx)
                    .compose(c -> c.preparedQuery(sqlCommand).execute(parameters))
                    .onFailure(ex -> log.error("Failed to execute prepared query: " + sqlCommand, ex));
        } catch (Exception ex) {
            return Future.failedFuture(ex);
        }
    }

    protected Future<Integer> prepareAndExecuteBatch(CommandContext ccx, String sql, List<Tuple> parameters, int maxBatchSize) {
        return beginTransaction(ccx)
                .compose(tcx -> {
                    try {
                        PreparedQuery<RowSet<Row>> preparedQuery = tcx.getConnection().preparedQuery(sql);
                        return executeBatches(preparedQuery, parameters, maxBatchSize).compose(tcx::commit)
                                .onFailure(ex -> log.error("Failed to execute batch: {}", sql))
                                .onComplete(tcx::close);
                    } catch (Exception ex) {
                        log.error("Failed to prepare and execute batch: {}", sql);
                        return tcx.close().compose(v -> Future.failedFuture(ex));
                    }
                });

    }

    public Future<Integer> executeBatches(PreparedQuery<RowSet<Row>> preparedQuery, List<Tuple> tuples, int batchSize) {

        if (tuples.size() > batchSize) {
            int batchCount = tuples.size() / batchSize;
            int r = tuples.size() % batchSize;
            if (r > 0) {
                batchCount++;
            }
            List<Future> futures = new ArrayList<>();
            for (int i = 0; i < batchCount; i++) {
                int beginIndex = i * batchSize;
                int endIndex = Math.min(beginIndex + batchSize - 1, tuples.size() - 1);
                List<Tuple> batchTuples = new ArrayList<>();
                for (int j = beginIndex; j <= endIndex; j++) {
                    batchTuples.add(tuples.get(j));
                }
                futures.add(preparedQuery.executeBatch(batchTuples));
            }
            return CompositeFuture.all(futures).map(res -> tuples.size());

        } else {
            return preparedQuery.executeBatch(tuples).map(res -> tuples.size());
        }
    }

    protected Future<Integer> executePreparedCommand(CommandContext ccx, SqlCommand sqlCommand, Tuple[] parameters) {
        return getClient(ccx).compose(clientForContext -> {
            int numberOfCombinedCommands = sqlCommand.numberOfCommands();
            if (numberOfCombinedCommands != parameters.length) {
                return Future.failedFuture("Number of commands does not match number of parameters (number of commands = "
                        + numberOfCombinedCommands + ", number of parameters = " + parameters.length);
            }
            AtomicInteger modificationsCount = new AtomicInteger();
            Future<Integer> future = Future.succeededFuture(0);
            for (int i = 0; i < numberOfCombinedCommands; i++) {
                String sqlForClient = getSqlForClient(i, clientForContext, ccx, sqlCommand);
                int parameterIndex = i;
                future = future
                        .compose(x -> executePreparedCommand(ccx, sqlForClient, parameters[parameterIndex]))
                        .map(c -> {
                            if (c != null) {
                                modificationsCount.addAndGet(c);
                            }
                            return c;
                        });
            }
            return future.map(r -> modificationsCount.get());
        });
    }

    protected Integer determineNumberOfModifications(List<Future> futures) {
        int numberOfModifications = -1;
        for (int i = 0; i < futures.size(); i++) {
            Integer number = (Integer) futures.get(0).result();
            if (number != null && number > numberOfModifications) {
                numberOfModifications = number;
            }
        }
        return numberOfModifications >= 0 ? numberOfModifications : null;
    }

    protected Future<Integer> executePreparedCommand(CommandContext ccx, SqlCommand sqlCommand, Tuple parameters) {
        return getSqlForClient(ccx, sqlCommand).compose(sql -> executePreparedCommand(ccx, sql, parameters));
    }

    protected Future<String> getSqlForClient(CommandContext ccx, SqlCommand sqlCommand) {
        return getClient(ccx)  // TODO refactore to avoid the need to get the client later again
                .map(clientForContext -> getSqlForClient(0, clientForContext, ccx, sqlCommand));
    }

    protected String getSqlForClient(int commandIndex, SqlClient clientForContext, CommandContext ccx, SqlCommand sqlCommand) {
        return sqlCommand.sqlForContext(commandIndex, this, client, ccx);
    }

    protected Future<Integer> executePreparedCommand(CommandContext ccx, String sqlCommand, Tuple parameters) {
        return executePreparedQuery(ccx, sqlCommand, parameters)
                .map(rowSet -> {
                    int created = rowSet != null ? rowSet.rowCount() : -1;
                    if (created <= 0) {
                        log.warn("Returned {} rows are created by command: '{}'", created, sqlCommand);
                    }
                    return created;
                })
                .onFailure(ex -> log.error("Failed to execute command " + sqlCommand, ex));
    }

    @Deprecated
    protected <T> AsyncReadStream<T> readAsStream(CommandContext ccx, String sqlCommand, Tuple parameters,
                                                  QuerySpecification querySpecification, Function<Row, T> mapping) {
        if (parameters == null || parameters.size() == 0) {  // ensure not to use static EMPTY
            parameters = new ArrayTuple(0);
        }
        int parameterIndex = parameters.size() + 1;
        if (querySpecification.isSorted()) {
            sqlCommand += " ORDER BY " + querySpecification.getSortBy() + " " + sortOrder(querySpecification);
        }
        if (querySpecification.isPaged()) {
            parameters.addInteger(querySpecification.getPageSize());
            parameters.addLong(offset(querySpecification.getPage(), querySpecification.getPageSize()));
            sqlCommand += " LIMIT $" + parameterIndex++ + " OFFSET $" + parameterIndex;
        }
        if (!querySpecification.isPaged() || querySpecification.getPageSize() > READ_ONCE_LIMIT) {
            return readAsStream(ccx, sqlCommand, parameters, mapping);
        } else {
            return readOnceAsStream(ccx, sqlCommand, parameters, mapping);
        }
    }

    protected <T> AsyncReadStream<T> readAsStream(CommandContext ccx, SqlCommand sqlCommand, Tuple parameters,
                                                  QuerySpecification querySpecification, Function<Row, T> mapping) {
        if (parameters == null || parameters.size() == 0) {  // ensure not to use static EMPTY
            parameters = new ArrayTuple(0);
        }
        String sqlCommandPostfix = "";
        int parameterIndex = parameters.size() + 1;
        if (querySpecification.isSorted()) {
            sqlCommandPostfix += " ORDER BY " + querySpecification.getSortBy() + " " + sortOrder(querySpecification);
        }
        if (querySpecification.isPaged()) {
            parameters.addInteger(querySpecification.getPageSize());
            parameters.addLong(offset(querySpecification.getPage(), querySpecification.getPageSize()));
            sqlCommandPostfix += " LIMIT $" + parameterIndex++ + " OFFSET $" + parameterIndex;
        }
        if (!querySpecification.isPaged() || querySpecification.getPageSize() > READ_ONCE_LIMIT) {
            return readAsStream(ccx, sqlCommand, sqlCommandPostfix, parameters, mapping);
        } else {
            return readOnceAsStream(ccx, sqlCommand, sqlCommandPostfix, parameters, mapping);
        }
    }

    protected String sortOrder(QuerySpecification querySpecification) {
        if (querySpecification.getDescending()) {
            return "DESC";
        } else {
            return "ASC";
        }
    }

    protected long offset(int page, int pageSize) {
        return (page - 1L) * pageSize;
    }

    @Deprecated
    protected <T> AsyncReadStream<T> readAsStream(CommandContext ccx, String sqlCommand, Function<Row, T> mapping) {
        return readAsStream(ccx, sqlCommand, null, mapping);
    }

    protected <T> AsyncReadStream<T> readAsStream(CommandContext ccx, SqlCommand sqlCommand, String sqlCommandPostfix,
                                                  Function<Row, T> mapping) {
        return readAsStream(ccx, sqlCommand, sqlCommandPostfix, null, mapping);
    }

    @Deprecated
    protected <T> AsyncReadStream<T> readAsStream(CommandContext ccx, String sqlCommand, Tuple parameters,
                                                  Function<Row, T> mapping) {
        return readAsStream(ccx, sqlCommand, parameters, mapping, -1);
    }

    protected <T> AsyncReadStream<T> readAsStream(CommandContext ccx, SqlCommand sqlCommand, String sqlCommandPostfix,
                                                  Tuple parameters, Function<Row, T> mapping) {
        return readAsStream(ccx, sqlCommand, sqlCommandPostfix, parameters, mapping, -1);
    }

    @Deprecated
    protected <T> AsyncReadStream<T> readAsStream(CommandContext ccx, String sqlCommand, Tuple parameters,
                                                  Function<Row, T> mapping, int fetch) {
        log.debug("Read as stream with cursor: {}", sqlCommand);
        MappedRowReadStream<T> readStream = new MappedRowReadStream<>(fetch, parameters, mapping);

        beginTransaction(ccx)
                .compose(tcx -> tcx.getConnection()
                        .prepare(sqlCommand)
                        .map(pg -> readStream.createRowStream(tcx.getConnection(), pg, tcx.getTransaction()))
                        .onFailure(readStream::fail)
                        .onFailure(ex -> tcx.close())

                ).onFailure(ex -> log.error("Failed to create ReadStream", ex));
        return readStream;
    }

    protected <T> AsyncReadStream<T> readAsStream(CommandContext ccx, SqlCommand sqlCommand, String sqlCommandPostfix,
                                                  Tuple parameters, Function<Row, T> mapping, int fetch) {
        log.debug("Read as stream with cursor: {}", sqlCommand);
        MappedRowReadStream<T> readStream = new MappedRowReadStream<>(fetch, parameters, mapping);

        beginTransaction(ccx)
                .compose(tcx -> {
                    SqlConnection c = tcx.getConnection();
                    String sql = sqlCommand.sqlForContext(this, c, ccx);
                    return c.prepare(sql + sqlCommandPostfix)
                            // .prepare(sqlCommand)
                            .map(pg -> readStream.createRowStream(tcx.getConnection(), pg, tcx.getTransaction()))
                            .onFailure(ex -> {
                                readStream.fail(ex);
                                tcx.close();
                            });
                })
                .onFailure(ex -> log.error("Failed to create ReadStream", ex));
        return readStream;
    }

    @Deprecated
    protected <T> AsyncReadStream<T> readOnceAsStream(CommandContext ccx, String sqlCommand, Tuple parameters,
                                                      Function<Row, T> mapping) {
        log.debug("Read once as stream: {}", sqlCommand);
        IterableReadStream<T> asyncReadStream = new IterableReadStream<>();
        getClient(ccx)
                .compose(c -> c.preparedQuery(sqlCommand).execute(parameters))
                .map(rows -> asyncReadStream.setIterable(rows, mapping))
                .onFailure(ex -> log.error("Failed to read once as stream", ex));
        return asyncReadStream;
    }

    protected <T> AsyncReadStream<T> readOnceAsStream(CommandContext ccx, SqlCommand sqlCommand, String sqlCommandPostfix,
                                                      Tuple parameters, Function<Row, T> mapping) {
        log.debug("Read once as stream: {}", sqlCommand);
        IterableReadStream<T> asyncReadStream = new IterableReadStream<>();
        getClient(ccx)
                .compose(c -> {
                    String sql = sqlCommand.sqlForContext(this, c, ccx);
                    return c.preparedQuery(sql + sqlCommandPostfix).execute(parameters);
                })
                .map(rows -> {
                    if (rows != null) {
                        asyncReadStream.setIterable(rows, mapping);
                    } else {
                        asyncReadStream.setIterable(Collections.emptySet(), mapping);
                    }
                    return rows;
                })
                .onFailure(ex -> log.error("Failed to read once as stream", ex));
        return asyncReadStream;
    }

    protected <T> Future<List<T>> readAsList(CommandContext ccx, String sqlCommand, Tuple parameters, Function<Row, T> mapping) {
        log.debug("Read as list: {}", sqlCommand);
        Promise<List<T>> promise = Promise.promise();
        List<T> list = new ArrayList<>();
        getClient(ccx)
                .compose(c -> c.preparedQuery(sqlCommand).mapping(mapping).execute(parameters))
                .map(rows -> {
                    for (T t : rows) {
                        list.add(t);
                    }
                    promise.complete(list);
                    return rows;
                })
                .onFailure(ex -> {
                    log.error("Failed to read as list", ex);
                    promise.fail(ex);
                });
        return promise.future();
    }

    public Future<Long> count(CommandContext ccx, String sqlCommand) {
        return count(ccx, sqlCommand, ArrayTuple.EMPTY);
    }

    public Future<Long> count(CommandContext ccx, String sqlCommand, Tuple parameters) {
        log.info("Count: " + sqlCommand);
        return findOneAndMap(ccx, sqlCommand, parameters, row -> row.getLong(0));
    }

    protected <R> Future<R> findOneAndMap(CommandContext ccx, SqlCommand sqlCommand, Tuple parameters, Function<Row, R> mapping) {
        return getSqlForClient(ccx, sqlCommand).compose(sql -> findOneAndMap(ccx, sql, parameters, mapping));
    }

    /**
     * Execute the sql query and maps the first row, in case multiple rows are found an exception is thrown.
     * 
     * @param <R>
     * @param sqlQuery
     * @param parameters
     * @param mapping
     * @return
     */
    protected <R> Future<R> findOneAndMap(CommandContext ccx, String sqlQuery, Tuple parameters, Function<Row, R> mapping) {
        if (parameters == null) {
            parameters = ArrayTuple.EMPTY;
        }
        return executePreparedQuery(ccx, sqlQuery, parameters)
                .map(rows -> mapSingle(rows, mapping, true));
    }


    protected <R> Future<R> findExactlyOneAndMap(CommandContext ccx, SqlCommand sqlCommand, Tuple parameters,
                                                 Function<Row, R> mapping) {
        return getSqlForClient(ccx, sqlCommand).compose(sql -> findExactlyOneAndMap(ccx, sql, parameters, mapping));
    }

    /**
     * Execute the sql query and maps the first row, in case not exactly one row is found an exception is thrown.
     * 
     * @param <R>
     * @param sqlQuery
     * @param parameters
     * @param mapping
     * @return
     */
    protected <R> Future<R> findExactlyOneAndMap(CommandContext ccx, String sqlQuery, Tuple parameters,
                                                 Function<Row, R> mapping) {
        return executePreparedQuery(ccx, sqlQuery, parameters)
                .map(rows -> mapSingle(rows, mapping, false))
                .onFailure(ex -> log.error("Failed to find exactly one: " + sqlQuery, ex));
    }

    protected <T, R> R mapSingle(RowSet<T> rows, Function<T, R> mapping, boolean acceptNotFound) {
        if (rows == null || rows.size() == 0) {
            if (acceptNotFound) {
                return null;
            } else {
                throw new NotFoundException("No rows found");
            }
        } else if (rows.size() == 1) {
            T row = rows.iterator().next();
            return mapping.apply(row);
        } else {
            throw new RuntimeException("Found multiple rows");
        }
    }


    protected String contains(String substring) {
        return "%" + substring + "%";
    }

    protected SqlCommand createSqlCommand(String sql) {
        return new SqlCommand(sql);
    }

    protected SqlCommand createSqlCommand(String... sql) {
        return new SqlCommand(sql);
    }

    protected SqlCommand createSqlCommand(SqlCommand... parts) {
        return new SqlCommand(parts);
    }

}
