package org.iworkz.genesis.vertx.common.persistence;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.function.Predicate;

import javax.inject.Inject;
import javax.inject.Singleton;

import org.iworkz.genesis.vertx.common.registry.GenesisRegistry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.Future;
import io.vertx.core.buffer.Buffer;
import io.vertx.sqlclient.Pool;


@Singleton
public class DatabaseSetup {
	
	private static final Logger log = LoggerFactory.getLogger(DatabaseSetup.class);

	private Set<AbstractDao<?>> registeredDaos;

	@Inject
	private GenesisRegistry commonRegistry;

	public Future<Integer> createSchemaOfDaos(Pool pool, String dbms, Predicate<AbstractDao<?>> daoFilter) {
		return applySchemaOfDaos(pool, dbms, SchemaContributionType.CREATE, daoFilter);
	}

	public Future<Integer> dropSchemaOfDaos(Pool pool, String dbms, Predicate<AbstractDao<?>> daoFilter) {
		return applySchemaOfDaos(pool, dbms, SchemaContributionType.DROP, daoFilter);
	}

	protected Future<Integer> applySchemaOfDaos(Pool pool, String dbms, SchemaContributionType schemaContributionType,
			Predicate<AbstractDao<?>> daoFilter) {

		Map<String, AbstractDao<?>> schemaContributionIds = collectSchemaContributionIds(dbms, daoFilter);
		return readSchemaContributions(dbms, schemaContributionType, schemaContributionIds)
				.map(this::mergeContributions)
				.compose(buffer -> {
					if (buffer.length() > 0) {
						log.info(buffer);
						return applySchemaDefinition(pool, buffer)
								.onFailure(cause -> log.error("Failed to apply schema contribution: {}", buffer, cause));
					} else {
						log.warn("No schema contributions from registered DAO packages found");
						return Future.succeededFuture(0);
					}
				});
	}
	
    protected String mergeContributions(List<Buffer> contributions) {
    	StringBuilder buffer = new StringBuilder();
        buffer.append("\r\n\r\n");
        for (Buffer contribution : contributions) {
            if (contribution != null) {
                buffer.append(contribution);
            }
        }
        return buffer.toString();
    }
    
    protected Future<Integer> applySchemaDefinition(Pool pool, String schemaDefintion) {
        return pool.query(schemaDefintion).execute().map(rowSet -> {
            if (rowSet != null) {
                return rowSet.size();
            } else {
                return 0;
            }
        });
    }

	protected Map<String, AbstractDao<?>> collectSchemaContributionIds(String dbms,
			Predicate<AbstractDao<?>> daoFilter) {

		Map<AbstractDao<?>, Set<Class<?>>> usesMap = new LinkedHashMap<>();
		for (AbstractDao<?> dao : getRegisteredDaos()) {
			Set<Class<?>> usedClasses = getClassesUsedByEntity(dao);
			usesMap.put(dao, usedClasses);
		}

		AbstractDao<?>[] sortedDaoArray = getRegisteredDaos().toArray(new AbstractDao[0]);
		for (int i = 0; i < sortedDaoArray.length - 1; i++) {
			for (int j = i + 1; j < sortedDaoArray.length; j++) {
				AbstractDao<?> dao1 = sortedDaoArray[i];
				AbstractDao<?> dao2 = sortedDaoArray[j];
				boolean compareResult = isBefore(dao1, dao2, usesMap);
				if (compareResult) {
					sortedDaoArray[i] = dao2;
					sortedDaoArray[j] = dao1;
				}
			}
		}
		List<AbstractDao<?>> sortedDaos = Arrays.asList(sortedDaoArray);

//        var sortedDaos = new ArrayList<AbstractDao<?>>(getRegisteredDaos());
//        sortedDaos.sort((dao1, dao2) -> compareDaos(dao1, dao2, usesMap));

		Map<String, AbstractDao<?>> additionalIds = new LinkedHashMap<>();
		Map<String, AbstractDao<?>> collectedIds = new LinkedHashMap<>();
		for (AbstractDao<?> dao : sortedDaos) {

			if (daoFilter == null || daoFilter.test(dao)) {
				String[] schemaContributionIdsOfDao = dao.getSchemaContributionIds(dbms);
				if (schemaContributionIdsOfDao != null) {
					for (int i = 0; i < schemaContributionIdsOfDao.length; i++) {
						String id = schemaContributionIdsOfDao[i];
						if (i == 0) {
							collectedIds.put(id, dao);
						} else {
							additionalIds.put(id, dao); // additional ones at the end
						}
					}
				}
			}
		}
		collectedIds.putAll(additionalIds);

		return collectedIds;
	}
	
	protected Future<List<Buffer>> readSchemaContributions(String dbms, SchemaContributionType type,
			Map<String, AbstractDao<?>> schemaContributionIds) {
		List<Buffer> contributions = new ArrayList<>();
		Future<Boolean> future = Future.succeededFuture(true);
		for (Entry<String, AbstractDao<?>> contributionEntry : schemaContributionIds.entrySet()) {
			future = future.compose(b -> {
				Future<Buffer> contribution = contributionEntry.getValue().getSchemaContribution(dbms, type,
						contributionEntry.getKey());
				return contribution.map(c -> contributions.add(c));
			});

		}
		return future.map(contributions);
	}

	protected Set<Class<?>> getClassesUsedByEntity(AbstractDao<?> dao) {
		Set<Class<?>> usedClasses = new HashSet<>();
		Class<?> entityClass = dao.getEntityClass();
		if (entityClass != null) {
			Class<?> superClass = entityClass.getSuperclass();
			if (superClass != null && Object.class != superClass && !Modifier.isAbstract(superClass.getModifiers())) {
				usedClasses.add(entityClass.getSuperclass());
			}
			for (Field field : entityClass.getDeclaredFields()) {
				Class<?> fieldType = field.getType();
				if (GenesisEntity.class.isAssignableFrom(fieldType)) {
					usedClasses.add(fieldType);
				}
			}
		}
		return usedClasses;
	}

	protected boolean isBefore(AbstractDao<?> dao1, AbstractDao<?> dao2, Map<AbstractDao<?>, Set<Class<?>>> usesMap) {
		Class<?> entityClass1 = dao1.getEntityClass();
		Class<?> entityClass2 = dao2.getEntityClass();
		Set<Class<?>> usedByDao1 = usesMap.get(dao1);
		Set<Class<?>> usedByDao2 = usesMap.get(dao2);
		if (usedByDao1.contains(entityClass2)) {
			return true;
		} else if (usedByDao2.contains(entityClass1)) {
			return false;
		} else {
			return dao1.getClass().getCanonicalName().compareTo(dao2.getClass().getCanonicalName()) > 1;
		}
	}

	public Set<AbstractDao<?>> getRegisteredDaos() {
		if (registeredDaos == null) {
			registeredDaos = new LinkedHashSet<>();
			for (GenesisDao<?> dao : commonRegistry.getDaos()) {
				if (dao instanceof AbstractDao) {
					registeredDaos.add((AbstractDao<?>) dao);
				}
			}
		}
		return registeredDaos;
	}

}
