package org.iworkz.genesis.vertx.common.persistence;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.function.Supplier;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.iworkz.genesis.vertx.common.context.CommandContext;

import io.vertx.sqlclient.SqlClient;


public class SqlCommand {

    private final String[] rawSql;
    private String[] sql;
    private Map<String, Supplier<String>> variables;

    protected static final String VARIABLE_REGEX = "\\$\\{[^\\}]*\\}";
    protected static final Pattern pattern = Pattern.compile(VARIABLE_REGEX);

    public SqlCommand(String rawSql) {
        this.rawSql = new String[1];
        this.rawSql[0] = rawSql;
    }

    public SqlCommand(String... rawSql) {
        this.rawSql = rawSql;
    }

    public SqlCommand(SqlCommand... parts) {
        if (parts != null) {
            this.rawSql = new String[1];
            StringBuilder buffer = new StringBuilder();
            for (SqlCommand part : parts) {
                buffer.append(part.getRawSql(0));
            }
            this.rawSql[0] = buffer.toString();
        } else {
            this.rawSql = null;
        }
    }

    public String getRawSql(int commandIndex) {
        if (rawSql != null) {
            return rawSql[commandIndex];
        } else {
            return null;
        }
    }

    public int numberOfCommands() {
        return rawSql != null ? rawSql.length : 0;
    }

    public <T extends GenesisEntity> String sqlForContext(AbstractDao<T> dao, SqlClient client,
                                                          CommandContext ccx) {
        return sqlForContext(0, dao, client, ccx);
    }

    public <T extends GenesisEntity> String sqlForContext(int commandIndex, AbstractDao<T> dao, SqlClient client,
                                                          CommandContext ccx) {
        if (sql == null) {
            sql = new String[rawSql.length];
            for (int i = 0; i < rawSql.length; i++) {
                sql[i] = preProcess(i, dao, client, ccx);
            }
        }
        return sql[commandIndex];
    }

    public SqlCommand defineVariable(String name, Supplier<String> supplier) {
        if (variables == null) {
            variables = new LinkedHashMap<>();
        }
        variables.put(name, supplier);
        return this;
    }

    protected <T extends GenesisEntity> String preProcess(int commandIndex, AbstractDao<T> dao, SqlClient client,
                                                          CommandContext ccx) {
        String sqlWithReplacedVariables = replaceVariables(getRawSql(commandIndex), ccx);
        String sqlWithReplacedStatementParameters = replaceStatementParameters(sqlWithReplacedVariables, ccx);
        if (isJdbcClient(client, ccx)) {
            return toQuestionMarkParameters(sqlWithReplacedStatementParameters);
        } else {
            return sqlWithReplacedStatementParameters;
        }
    }

    protected String replaceVariables(String sourceSql, CommandContext ccx) {
        if (variables != null) {
            String replacedSql = sourceSql;
            for (Entry<String, Supplier<String>> variable : variables.entrySet()) {
                String name = variable.getKey();
                String value = variable.getValue().get();
                replacedSql = replacedSql.replace("${" + name + "}", value);
            }
            return replacedSql;
        } else {
            return sourceSql;
        }
    }

    protected String replaceStatementParameters(String sourceSql, CommandContext ccx) {
        Matcher matcher = pattern.matcher(sourceSql);
        StringBuilder buffer = new StringBuilder();
        int parameterNumber = 0;
        int currentIndex = 0;
        while (matcher.find()) {
            int start = matcher.start();
            int end = matcher.end();
            String part = sourceSql.substring(currentIndex, start);
            buffer.append(part);
            buffer.append('$');
            buffer.append(parameterNumber++);
            currentIndex = end;
        }
        String remaining = sourceSql.substring(currentIndex);
        buffer.append(remaining);
        return buffer.toString();
    }

    protected boolean isJdbcClient(SqlClient client, CommandContext ccx) {
        return true;
    }

    protected String toQuestionMarkParameters(String sourceSql) {
        return sourceSql.replaceAll("[$]\\d*", "?");
    }

}
