package org.iworkz.genesis.vertx.common.queue;

import java.util.HashMap;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicInteger;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.AsyncResult;
import io.vertx.core.Future;
import io.vertx.core.Promise;
import io.vertx.core.Vertx;

public abstract class AbstractAsyncQueue<T> {

    private static final Logger log = LoggerFactory.getLogger(AbstractAsyncQueue.class);
    private static final Long DEFAULT_PROCESSING_TIMEOUT = 30000L;

    private final Vertx vertx;
    private final int maxParallelCount;
    private final int maxRetries;
    private final long processingTimeout;

    private AtomicInteger totalCount = new AtomicInteger();
    private AtomicInteger finishedCount = new AtomicInteger();
    private AtomicInteger processingCount = new AtomicInteger();

    private boolean failed = false;
    private boolean finished = false;
    private Promise<Void> promise = Promise.promise();

    private Queue<T> queue = new ConcurrentLinkedQueue<>();
    private Map<T, Integer> retryMap = new HashMap<>();

    protected AbstractAsyncQueue(Vertx vertx, int maxParallelCount, int maxRetries) {
        this(vertx, maxParallelCount, maxRetries, DEFAULT_PROCESSING_TIMEOUT);
    }

    protected AbstractAsyncQueue(Vertx vertx, int maxParallelCount, int maxRetries, long processingTimeout) {
        this.maxParallelCount = maxParallelCount;
        this.maxRetries = maxRetries;
        this.vertx = vertx;
        this.processingTimeout = processingTimeout;
    }

    public void setFinished() {
        finished = true;
        continueProcessing();
    }

    public Future<Void> done() {
        return promise.future();
    }

    public void add(T data) {
        if (!failed) {
            totalCount.incrementAndGet();
            addItem(data);
            continueProcessing();
        }
    }

    protected void addItem(T item) {
        queue.add(item);
    }

    protected void handleNextItem(T item) {
        if (item != null) {
            try {
                Promise<Void> handlingPromise = Promise.promise();
                long timerId = createProcessingTimeoutTimer(handlingPromise);
                processAsync(item).onComplete(res -> {
                    if (timerId > -1L) {
                        vertx.cancelTimer(timerId);
                    }
                    if (res.succeeded()) {
                        handlingPromise.tryComplete(res.result());
                    } else {
                        handlingPromise.tryFail(res.cause());
                    }
                });
                handlingPromise.future().onComplete(res -> itemFinished(item, res));
            } catch (Exception ex) {
                log.error("Exception occured during async processing: {}", ex.getMessage(), ex);
                itemFinished(item, Future.failedFuture(ex));
            }
        }
    }

    protected long createProcessingTimeoutTimer(Promise<Void> handlingPromise) {
        if (processingTimeout > 0) {
            return vertx.setTimer(processingTimeout, l -> handlingPromise.tryFail(("Item processing timed out")));
        } else {
            return -1L;
        }
    }

    protected T retrieveNextItem() {
        try {
            if (!failed) {
                T item = queue.poll();
                if (item != null) {
                    processingCount.incrementAndGet();
                }
                return item;
            }
        } catch (Exception ex) {
            failQueue(ex);
        }
        return null;
    }

    protected void itemFinished(T item, AsyncResult<Void> res) {
        try {
            processingCount.decrementAndGet();
            if (!failed) {
                if (res.failed()) {
                    if (failedFinallyAfterRetries(item)) {
                        log.info("Failed to process item, after {} retries", maxRetries, res.cause());
                        failQueue(res.cause());
                    } else {
                        log.info("Add to queue for retry");
                        addItem(item);         // add again to queue (at the end)
                        continueProcessing();  // continue with by checking queue for next item
                    }
                } else {
                    finishedCount.incrementAndGet();
                    continueProcessing();
                }
            }
        } catch (Exception ex) {
            log.error("Failed to finish async item processing", ex);
        }
    }

    protected void continueProcessing() {
        vertx.runOnContext(v -> {
            try {
                checkQueue();
            } catch (Exception ex) {
                failQueue(ex);
            }
        });
    }

    public void checkQueue() {
        if (failed) {
            log.debug("Skip queue-check because already failed");
        } else {
            int currentlyProcessingCount = processingCount.get();
            boolean readyForNextItem = currentlyProcessingCount < maxParallelCount;
            if (finished && currentlyProcessingCount <= 0 && queue.isEmpty()) {
                log.debug("Item finished successful.");
                promise.tryComplete();
            } else if (readyForNextItem) {
                T item = retrieveNextItem();
                if (item != null) {
                    if (log.isDebugEnabled()) {
                        log.debug("Start processing {} of {} (currently processing: {})",
                                finishedCount.get() + currentlyProcessingCount + 1, totalCount.get(), currentlyProcessingCount);
                    }
                    handleNextItem(item);
                } else {
                    log.info(
                            "Ready for next item and queue is empty: finished={}, processingCount={}, totalCount={}, queueSize={}, finishedCount={}, retryCount={}",
                            finished, currentlyProcessingCount, totalCount, queue.size(), finishedCount, retryMap.size());
                }
            }
        }
    }

    protected void failQueue(Throwable cause) {
        failed = true;
        log.info("Failed to process item, after {} retries", maxRetries, cause);
        if (promise.tryFail(cause)) {
            log.error("Queue done with fail", cause);
        }
    }

    protected boolean failedFinallyAfterRetries(T item) {
        synchronized (retryMap) {
            Integer passedRetries = retryMap.get(item);
            if (passedRetries == null) {
                passedRetries = 0;
            }
            passedRetries++;
            retryMap.put(item, passedRetries);
            return passedRetries >= maxRetries;
        }
    }

    protected abstract Future<Void> processAsync(T data);

}
