package org.iworkz.genesis.vertx.common.registry;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.PostConstruct;
import javax.inject.Inject;

import org.iworkz.genesis.Injector;
import org.iworkz.genesis.vertx.common.persistence.GenesisDao;
import org.iworkz.genesis.vertx.common.persistence.GenesisEntity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.github.classgraph.ClassGraph;
import io.github.classgraph.ClassInfo;
import io.github.classgraph.ScanResult;


public abstract class GenesisRegistry {

    private static final Logger log = LoggerFactory.getLogger(GenesisRegistry.class);

    private static final String[] DEFAULT_SUB_PACKAGES = { "entity", "dao" };

    private List<String> basePackages = new ArrayList<>();
    private Map<String, List<Class<? extends GenesisDao<?>>>> daoMap = new LinkedHashMap<>();
    private Set<Class<? extends GenesisDao<?>>> daoClasses = new LinkedHashSet<>();

    private Set<GenesisDao<? extends GenesisEntity>> daos = new LinkedHashSet<>();

    @Inject
    private Injector injector;


    @PostConstruct
    public void setup() {
        if (basePackages.isEmpty()) {
            log.warn("No packages registered in {}", getClass().getCanonicalName());
        }
        register();
        long beginScan = System.currentTimeMillis();
        scanPackages();
        createDaoInstances();
        long endScan = System.currentTimeMillis();
        log.info("Scanned packages in {} ms", endScan - beginScan);
    }

    public Collection<GenesisDao<? extends GenesisEntity>> getDaos() {
        return daos;
    }

    protected void createDaoInstances() {
        for (Class<? extends GenesisDao<?>> daoClass : daoClasses) {
            try {
                GenesisDao<?> dao = injector.getInstance(daoClass);
                daos.add(dao);
                log.info("Created DAO {}", daoClass.getCanonicalName());
            } catch (Exception ex) {
                log.error("Failed to create instance of DAO " + daoClass.getCanonicalName(), ex);
            }
        }
    }

    protected abstract void register();

    public void registerPackage(String pkg) {
        basePackages.add(pkg);
    }

    protected void registerDao(Class<? extends GenesisDao<?>> daoClass) {
        daos.add(injector.getInstance(daoClass));
    }

    public Set<String> getDaoPackages() {
        return daoMap.keySet();
    }

    protected void scanPackages() {

        try (ScanResult scanResult = createScanner().scan()) {

            for (ClassInfo entityClassInfo : scanResult.getAllClasses()) {

                if (entityClassInfo.isAbstract()) {
                    continue;
                }

                try {

                    Class<?> classInPackage = entityClassInfo.loadClass();

                    if (GenesisEntity.class.isAssignableFrom(classInPackage)) {
                        log.debug("Entity class: {}", entityClassInfo.getName());
                    } else if (GenesisDao.class.isAssignableFrom(classInPackage)) {
                        log.debug("Dao class: {}", entityClassInfo.getName());

                        Class<? extends GenesisDao<?>> daoClass = (Class<? extends GenesisDao<?>>) classInPackage;
                        String packageName = classInPackage.getPackage().getName();
                        List<Class<? extends GenesisDao<?>>> daoClassesOfPackage = daoMap.get(packageName);
                        if (daoClassesOfPackage == null) {
                            daoClassesOfPackage = new ArrayList<>();
                            daoMap.put(packageName, daoClassesOfPackage);
                        }
                        daoClassesOfPackage.add(daoClass);
                        daoClasses.add(daoClass);
                    } else {
                        log.trace("Class: {}", entityClassInfo.getName());
                    }


                } catch (Exception ex) {
                    throw new RuntimeException("Failed to load class: " + entityClassInfo.getName());
                }
            }
        }

    }

    protected ClassGraph createScanner() {
        return new ClassGraph().acceptPackagesNonRecursive(getPackagesToScan());
    }

    protected String[] getPackagesToScan() {
        List<String> packagesToScan = new ArrayList<>();
        for (String pkg : basePackages) {
            for (String subPackage : getSubPackages()) {
                packagesToScan.add(pkg + "." + subPackage);
            }
        }
        return packagesToScan.toArray(new String[packagesToScan.size()]);
    }

    protected String[] getSubPackages() {
        return DEFAULT_SUB_PACKAGES;
    }

}
