package org.iworkz.genesis.vertx.common.router;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Stream;

import org.iworkz.genesis.vertx.common.controller.RESTController;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.Vertx;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.http.HttpMethod;
import io.vertx.core.json.JsonObject;
import io.vertx.ext.web.Route;
import io.vertx.ext.web.Router;
import io.vertx.ext.web.handler.StaticHandler;
import io.vertx.ext.web.openapi.OpenAPIHolder;
import io.vertx.ext.web.openapi.impl.ContractEndpointHandler;

public class RESTRouterBuilderImpl implements RESTRouterBuilder {

    private static final Logger log = LoggerFactory.getLogger(RESTRouterBuilderImpl.class);

    private final Vertx vertx;
    private final String contractEndPoint;
    private final GenesisOpenAPIHolderImpl openapi;

    private Map<String, GenesisOperationImpl> operations;

    private Set<RESTController> restControllers = new HashSet<>();

    public RESTRouterBuilderImpl(Vertx vertx, GenesisOpenAPIHolderImpl openapi, String contractEndPoint) {
        this.vertx = vertx;
        this.contractEndPoint = contractEndPoint;
        this.openapi = openapi;
        initializeOperations();
    }

    protected void initializeOperations() {
        if (operations == null) {
            operations = new LinkedHashMap<>();
            openapi.solveIfNeeded(openapi.getOpenAPI().getJsonObject("paths")).forEach(pathEntry -> {
                if (pathEntry.getKey().startsWith("x-"))
                    return;
                JsonObject pathModel = openapi.solveIfNeeded((JsonObject) pathEntry.getValue());
                Stream.of(
                        "get", "put", "post", "delete", "options", "head", "patch", "trace")
                        .filter(pathModel::containsKey)
                        .forEach(verb -> {
                            JsonObject operationModel = openapi.solveIfNeeded(pathModel.getJsonObject(verb));
                            String operationId = operationModel.getString("operationId");
                            this.operations.put(
                                    operationId,
                                    new GenesisOperationImpl(
                                            operationId,
                                            HttpMethod.valueOf(verb.toUpperCase()),
                                            pathEntry.getKey(),
                                            operationModel,
                                            pathModel,
                                            openapi.getGenesisInitialScope(),
                                            openapi));
                        });
            });
        }
    }

    @Override
    public OpenAPIHolder getOpenAPI() {
        return openapi;
    }

    @Override
    public void addRESTController(RESTController controller) {
        restControllers.add(controller);
    }

    @Override
    public Router addRoutesTo(Router router) {
        for (RESTController controller : restControllers) {

            Map<String, List<OperationMapping>> operationsMap = new LinkedHashMap<>();
            for (RESTOperation controllerOperation : controller.getRESTOperations()) {
                GenesisOperationImpl operation = operations.get(controllerOperation.getOperationId());
                if (operation != null) {
                    addOperationMapping(operationsMap, operation, controllerOperation);
                } else {
                    log.error("REST Operation with id '{}' not routed", controllerOperation.getOperationId());
                }
            }

            List<String> operationPaths = new ArrayList<>(operationsMap.keySet());
            sortPaths(operationPaths);
            addSortedPaths(router, operationPaths, operationsMap);
        }

        if (contractEndPoint != null) {
            String swaggerUiEndpoint = contractEndPoint + "/ui";
            router.get(swaggerUiEndpoint + "*").handler(StaticHandler.create(swaggerUiEndpoint.substring(1)));
            router.get(contractEndPoint).handler(ContractEndpointHandler.create(this.openapi));
        }
        return router;
    }

    protected void addOperationMapping(Map<String, List<OperationMapping>> operationsMap,
                                       GenesisOperationImpl operation, RESTOperation controllerOperation) {

        OperationMapping operationMapping = new OperationMapping();
        operationMapping.setGenesisOperation(operation);
        operationMapping.setRestOperation(controllerOperation);

        String vertxPath = composeVertxPath(operation);
        List<OperationMapping> genesisOperations = operationsMap.computeIfAbsent(vertxPath, p -> new ArrayList<>());
        genesisOperations.add(operationMapping);
    }

    protected void sortPaths(List<String> operationPaths) {
        operationPaths.sort((s1, s2) -> s2.compareTo(s1));  // Descending in order to ensure colons first
    }

    protected void addSortedPaths(Router router, List<String> operationPaths, Map<String, List<OperationMapping>> operationsMap) {

        for (String vertxPath : operationPaths) {
            List<OperationMapping> operationMappings = operationsMap.get(vertxPath);
            for (OperationMapping operationMapping : operationMappings) {
                GenesisOperationImpl operation = operationMapping.getGenesisOperation();
                RESTOperation controllerOperation = operationMapping.getRestOperation();
                addHttpRoute(router, vertxPath, operation, controllerOperation);
                addEventBusRoute(vertxPath, operation, controllerOperation);

                log.info("Route added for '{}': method={}, path={}", controllerOperation.getOperationId(),
                        operation.getHttpMethod(), vertxPath);

            }
        }
    }

    protected String composeVertxPath(GenesisOperationImpl operation) {
        return operation.getOpenAPIPath().replace('{', ':').replace("}", "");
    }

    protected void addHttpRoute(Router router, String vertxPath, GenesisOperationImpl operation,
                                RESTOperation controllerOperation) {
        router.route(operation.getHttpMethod(), vertxPath)
                .handler(controllerOperation.getHttpHandler());
    }

    protected void addEventBusRoute(String vertxPath, GenesisOperationImpl operation, RESTOperation controllerOperation) {
        vertx.eventBus().<Buffer>consumer(operation.getOperationId()).handler(controllerOperation.getEventBusHandler());
    }

}
