package org.iworkz.genesis.vertx.common.stream;

import java.util.ArrayList;
import java.util.List;

import org.iworkz.common.exception.GenesisException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.Future;
import io.vertx.core.Handler;
import io.vertx.core.Promise;
import io.vertx.core.streams.ReadStream;

public abstract class AbstractAsyncReadStream<T> implements AsyncReadStream<T> {

    private static final Logger log = LoggerFactory.getLogger(AbstractAsyncReadStream.class);

    private Handler<T> targetItemHandler;
    private Handler<T> lastTargetItemHandler;

    private Handler<Void> endHandler;
    private Handler<Throwable> exceptionHandler;

    private List<Handler<T>> additionalHandlers;

    protected final Promise<AsyncReadStream<T>> readyPromise = Promise.promise();

    protected boolean failed = false;

    @Override
    public ReadStream<T> handler(Handler<T> handler) {
        if (targetItemHandler != null && handler == null) {
            log.debug("Clear target item handler in stream {}", this);
            if (targetItemHandler != null) {
                lastTargetItemHandler = targetItemHandler;
            }
        }
        this.targetItemHandler = handler;
        return this;
    }

    protected Handler<T> getTargetItemHandler() {
        return targetItemHandler;
    }

    @Override
    public AbstractAsyncReadStream<T> resume() {
        if (targetItemHandler == null && !isTargetItemHandlerCleared()) {
            log.error("Illegal state: target item handler needs to be set when stream is resumed");
            handleException(new IllegalStateException("Target item handler is not defined"));
        }
        return this;
    }

    protected boolean isTargetItemHandlerCleared() {
        return lastTargetItemHandler != null;
    }

    protected void invokeTargetItemHandler(T target) {
        Handler<T> handlerToInvoke = getTargetItemHandler();
        if (handlerToInvoke != null) {
            handlerToInvoke.handle(target);
            if (additionalHandlers != null) {
                for (Handler<T> additionalHandler : additionalHandlers) {
                    additionalHandler.handle(target);
                }
            }
        } else if (lastTargetItemHandler != null) {
            log.warn("Fall back to last known handler in stream = {}, item = {}", this, target);
            targetItemHandler = lastTargetItemHandler;
            invokeTargetItemHandler(target);  // try again and continue with fallback
        } else {
            log.error("Failed to handle item in stream {}, because handler is null, item = {}", this, target);
            throw new GenesisException("Stream handler is not defined");
        }
    }

    protected <E> void handleSourceItem(E source) {
        invokeTargetItemHandler((T) source);
    }

    @Override
    public AsyncReadStream<T> withItem(Handler<T> handler) {
        if (additionalHandlers == null) {
            additionalHandlers = new ArrayList<>();
        }
        additionalHandlers.add(handler);
        return this;
    }

    @Override
    public AsyncReadStream<T> endHandler(Handler<Void> endHandler) {
        this.endHandler = endHandler;
        return this;
    }

    @Override
    public ReadStream<T> exceptionHandler(Handler<Throwable> exceptionHandler) {
        this.exceptionHandler = exceptionHandler;
        return this;
    }

    public Handler<Void> getEndHandler() {
        return endHandler;
    }

    public Handler<Throwable> getExceptionHandler() {
        return exceptionHandler;
    }


    @Override
    public Future<AsyncReadStream<T>> available() {
        return readyPromise.future();
    }

    public void fail(Throwable cause) {
        if (readyPromise.future().isComplete()) {
            handleException(cause);
        }
        readyPromise.tryFail(cause);
    }

    protected void handleException(Throwable ex) {
        synchronized (this) {
            failed = true;
            if (getExceptionHandler() != null) {
                Handler<Throwable> currentExceptionHandler = getExceptionHandler();
                currentExceptionHandler.handle(ex);
                onException(ex)
                        .onFailure(cause -> log.error("Failed to handle exception", cause))
                        .eventually(v -> {
                            return Future.succeededFuture();
                        });
            } else {
                onException(ex);
            }
        }
    }

    protected void handleEnd(Void v) {
        synchronized (this) {
            if (!failed) {
                try {
                    finishItemHandling()
                            .onSuccess(f -> {
                                if (getEndHandler() != null) {
                                    Handler<Void> currentEndHandler = getEndHandler();
                                    currentEndHandler.handle(v);
                                    onEnd()
                                            .onFailure(cause -> log.error("Failed to handle end", cause))
                                            .eventually(v1 -> {
                                                return Future.succeededFuture();
                                            });
                                } else {
                                    onEnd();
                                }
                            })
                            .onFailure(cause -> handleException(cause));
                } catch (Exception ex) {
                    handleException(ex);
                }

            }
        }
    }

    protected Future<Void> finishItemHandling() {
        return Future.succeededFuture();
    }

    protected Future<Void> onEnd() {
        return Future.succeededFuture();
    }

    protected Future<Throwable> onException(Throwable ex) {
        log.error("Streaming exection occured", ex);
        return Future.succeededFuture(ex);
    }

}
