package org.iworkz.genesis.vertx.common.stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.Future;
import io.vertx.core.streams.ReadStream;
import io.vertx.core.streams.WriteStream;

public abstract class AbstractMappingStream<S, T> extends AbstractAsyncReadStream<T> {

    private static final Logger log = LoggerFactory.getLogger(AbstractMappingStream.class);

    private ReadStream<S> readStream;
    private boolean readStreamHandlerIsSet;


    public void mapStream(ReadStream<S> sourceStream) {
        if (sourceStream instanceof AsyncReadStream) {
            AsyncReadStream<S> asyncSourceStream = (AsyncReadStream<S>) sourceStream;
            asyncSourceStream.available()
                    .map(s -> {
                        mapStreamHandlers(asyncSourceStream);
                        readyPromise.complete(this);
                        return s;
                    })
                    .onFailure(this::fail);
        } else {
            mapStreamHandlers(sourceStream);
            readyPromise.complete(this);
        }
    }

    protected void mapStreamHandlers(ReadStream<S> sourceStream) {
        if (readStream != null) {
            throw new IllegalStateException("Source stream already set");
        }
        if (sourceStream == null) {
            throw new IllegalArgumentException("Source stream must not be null");
        }

        this.readStream = sourceStream;

        readStream.endHandler(this::handleEnd);
        readStream.exceptionHandler(this::handleException);

        /*
         * Do not set read stream handler already now because this would start streaming to early
         * (mapping handlers are usually set later).
         * The read stream handler is checked and set when resume is called.
         */

    }

    @Override
    public AbstractMappingStream<S, T> pause() {
        readStream.pause();
        return this;
    }

    @Override
    public AbstractMappingStream<S, T> resume() {
        checkAndSetReadStreamHandlerIfNotSet();
        super.resume();
        readStream.resume();
        return this;
    }

    protected void checkAndSetReadStreamHandlerIfNotSet() {
        if (!readStreamHandlerIsSet) {
            synchronized (this) {
                if (!readStreamHandlerIsSet) {
                    readStream.handler(this::invokeHandleItem);
                    readStreamHandlerIsSet = true;
                }
            }
        }
    }

    @Override
    public AbstractMappingStream<S, T> fetch(long amount) {
        readStream.fetch(amount);
        return this;
    }

    protected void checkStream() {
        if (readStream == null) {
            log.error("Mapped stream not available yet");
        }
    }

    @Override
    public Future<Void> pipeTo(WriteStream<T> dst) {
        return pipeTo(dst, true);
    }

    protected void invokeHandleItem(S source) {
        try {
            handleSourceItem(source);
        } catch (Exception ex) {
            handleException(ex);
        }
    }

    @Override
    protected abstract <E> void handleSourceItem(E source);

}
