package org.iworkz.genesis.vertx.common.stream;

import java.util.List;
import java.util.function.Function;

import org.iworkz.genesis.vertx.common.stream.aggregator.AbstractStreamAggregator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.AsyncResult;
import io.vertx.core.Future;
import io.vertx.core.Handler;
import io.vertx.core.streams.ReadStream;
import io.vertx.core.streams.WriteStream;
import io.vertx.core.streams.impl.PipeImpl;

public interface AsyncReadStream<T> extends ReadStream<T> {

    static final Logger log = LoggerFactory.getLogger(AsyncReadStream.class);

    default Future<AsyncReadStream<T>> available() {
        return Future.succeededFuture(this);
    }

    default Future<Void> forEach(Function<T, Future<Void>> handler) {
        return ReadStreamConsumer.forEach(this, handler);
    }

    default Future<T> head() {
        return toList().map(list -> {
            if (list.isEmpty()) {
                return null;
            } else {
                return list.get(0);
            }
        });
    }

    default Future<List<T>> toList() {
        return ReadStreamConsumer.toList(this);
    }

    @Override
    default Future<Void> pipeTo(WriteStream<T> dst) {
        return pipeTo(dst, true);
    }

    @Override
    default void pipeTo(WriteStream<T> dst, Handler<AsyncResult<Void>> handler) {
        available()
                .compose(v -> new PipeImpl<>(this).to(dst))
                .onComplete(result -> {
                    try {
                        handler.handle(result);
                    } catch (Exception ex) {
                        log.error("Pipe failed", ex);
                    }
                });
    }

    default Future<Void> pipeTo(WriteStream<T> dst, boolean enOnFailure) {
        return available().compose(v -> new PipeImpl<>(this).endOnFailure(enOnFailure).to(dst));
    }

    default <R> AsyncReadStream<R> map(Function<T, R> mappingFunction) {
        MappedReadStream<T, R> mappedReadStream = new MappedReadStream<>(mappingFunction);
        mappedReadStream.mapStream(this);
        return mappedReadStream;
    }

    default AsyncReadStream<T> afterHandling(Handler<T> handler) {
        return this;
    }

    default <I, R> AsyncReadStream<R> aggregate(AbstractStreamAggregator<T, I, R> resourceNodeAggregator) {
        AggregatingStream<T, R> aggregatedStream = new AggregatingStream<>(resourceNodeAggregator);
        aggregatedStream.mapStream(this);
        return aggregatedStream;
    }

    @Override
    AsyncReadStream<T> endHandler(Handler<Void> endHandler);

    AsyncReadStream<T> withItem(Handler<T> handler);


}
