package org.iworkz.genesis.vertx.common.stream;

import java.util.LinkedList;
import java.util.Queue;
import java.util.UUID;

import io.vertx.core.streams.ReadStream;

public class CompositeReadStream<T> extends AbstractAsyncReadStream<T> {

    private static final int DEFAULT_QUEUE_SIZE = 20;

    private final int queueSize;

    private long fetchAmount = Long.MAX_VALUE;

    private Queue<UUID> subscribedStreamIds = new LinkedList<>();

    private boolean membersPaused = true;
    private Queue<AsyncReadStream<T>> pendingStreams = new LinkedList<>();
    private Queue<AsyncReadStream<T>> queuedStreams = new LinkedList<>();

    public CompositeReadStream() {
        this(DEFAULT_QUEUE_SIZE);
    }

    public CompositeReadStream(int queueSize) {
        this.queueSize = queueSize;
    }

    public AsyncReadStream<T> addMemberStream(AsyncReadStream<T> memberStream) {
        UUID idSubscribed = subscribeMemberStream();
        return addMemberStream(idSubscribed, memberStream);
    }

    public AsyncReadStream<T> addMemberStream(UUID idSubscribed, AsyncReadStream<T> memberStream) {

        memberStream.available()
                .map(s -> {

                    boolean directlyPutToQueue = false;
                    synchronized (queuedStreams) {
                        if (queuedStreams.size() < queueSize) {
                            queuedStreams.add(memberStream);
                            directlyPutToQueue = true;
                            resumeMemberStream(memberStream);
                        }
                    }

                    synchronized (pendingStreams) {
                        if (!directlyPutToQueue) {
                            pendingStreams.add(memberStream);
                            memberStream.pause();
                        }
                        subscribedStreamIds.remove(idSubscribed);
                    }

                    readyPromise.tryComplete(this);
                    return s;
                })
                .onFailure(this::fail);

        memberStream.endHandler(v -> {
            AsyncReadStream<T> nextStream = null;
            synchronized (pendingStreams) {
                if (!pendingStreams.isEmpty()) {
                    nextStream = pendingStreams.remove();
                }
            }
            synchronized (queuedStreams) {
                queuedStreams.remove(memberStream);
                if (nextStream != null) {
                    queuedStreams.add(nextStream);
                    resumeMemberStream(nextStream);
                }
                if (subscribedStreamIds.isEmpty() && allEnded()) {
                    handleEnd(null);
                }
            }
        });
        memberStream.exceptionHandler(this::handleException);

        return this;
    }

    protected void resumeMemberStream(AsyncReadStream<T> memberStream) {
        if (!membersPaused) {
            memberStream.handler(this::invokeHandleItem);
            memberStream.resume();
        }
    }

    public UUID subscribeMemberStream() {
        UUID id = UUID.randomUUID();
        synchronized (pendingStreams) {
            subscribedStreamIds.add(id);
        }
        return id;
    }

    protected boolean allEnded() {
        return queuedStreams.isEmpty();
    }

    @Override
    public ReadStream<T> pause() {
        synchronized (queuedStreams) {
            membersPaused = true;
            for (AsyncReadStream<T> memberStream : queuedStreams) {
                memberStream.pause();
            }
        }
        return this;
    }

    @Override
    public CompositeReadStream<T> resume() {
        synchronized (queuedStreams) {
            super.resume();
            membersPaused = false;
            for (AsyncReadStream<T> memberStream : queuedStreams) {
                memberStream.handler(this::invokeHandleItem);
                memberStream.resume();
            }
        }
        return this;
    }

    @Override
    public ReadStream<T> fetch(long amount) {
        synchronized (queuedStreams) {
            if (amount > 0) {
                fetchAmount = amount;
            }
            for (AsyncReadStream<T> memberStream : queuedStreams) {
                memberStream.fetch(amount);
            }
        }
        return this;
    }

    protected synchronized void invokeHandleItem(T source) {
        try {
            fetchAmount--;
            handleSourceItem(source);
        } catch (Exception ex) {
            handleException(ex);
        }
    }

}
