package org.iworkz.genesis.vertx.common.stream;

import java.util.function.Function;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.Future;
import io.vertx.sqlclient.PreparedStatement;
import io.vertx.sqlclient.Row;
import io.vertx.sqlclient.SqlConnection;
import io.vertx.sqlclient.Transaction;
import io.vertx.sqlclient.Tuple;
import io.vertx.sqlclient.impl.ArrayTuple;

public class MappedRowReadStream<T> extends MappedReadStream<Row, T> {

    private static final Logger log = LoggerFactory.getLogger(MappedRowReadStream.class);

    private static final int DEFAULT_FETCH = 50;

    private final int fetch;
    private final Tuple parameters;

    private SqlConnection conn;
    private Transaction tx;


    public MappedRowReadStream(Function<Row, T> mapping) {
        this(-1, mapping);
    }

    public MappedRowReadStream(int fetch, Function<Row, T> mapping) {
        this(fetch, null, mapping);
    }

    public MappedRowReadStream(int fetch, Tuple parameters, Function<Row, T> mapping) {
        super(mapping);
        this.fetch = fetch > 0 ? fetch : DEFAULT_FETCH;
        this.parameters = parameters != null ? parameters : ArrayTuple.EMPTY;
    }

    @Override
    public Future<Void> onEnd() {
        return tx.commit()
                .recover(cause -> {
                    log.error("Failed to commit transaction", cause);
                    return Future.succeededFuture();
                })
                .compose(v -> conn.close())
                .recover(cause -> {
                    log.error("Failed to close connection after commit", cause);
                    return Future.succeededFuture();
                });
    }

    @Override
    public Future<Throwable> onException(Throwable ex) {
        log.error("MappedRowReadStream failed", ex);
        if (tx != null) {
            return tx.rollback()
                    .recover(cause -> {
                        log.error("Failed to rollback transaction", cause);
                        return Future.succeededFuture();
                    })
                    .compose(v -> conn.close())
                    .recover(cause -> {
                        log.error("Failed to close connection after rollback", cause);
                        return Future.succeededFuture();
                    })
                    .compose(r -> Future.succeededFuture(ex));
        } else if (conn != null) {
            return conn.close()
                    .recover(cause -> {
                        log.error("Failed to close connection", cause);
                        return Future.succeededFuture();
                    })
                    .compose(r -> Future.succeededFuture(ex));
        } else {
            return Future.succeededFuture(ex);
        }
    }

    public MappedRowReadStream<T> createRowStream(SqlConnection conn, PreparedStatement pg, Transaction tx) {
        this.conn = conn;
        this.tx = tx;
        mapStream(pg.createStream(fetch, parameters));
        return this;
    }

}
