package org.iworkz.genesis.vertx.common.stream;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.AsyncResult;
import io.vertx.core.Future;
import io.vertx.core.Handler;
import io.vertx.core.Promise;
import io.vertx.core.streams.WriteStream;

public class ReadStreamConsumer<T> implements WriteStream<T> {

    private static final Logger log = LoggerFactory.getLogger(ReadStreamConsumer.class);

    private final Function<T, Future<Void>> writeHandler;
    private Handler<Throwable> exceptionHandler;
    private Handler<AsyncResult<Void>> endHandler;

    private Handler<Void> drainHandler;
    private int maxQueueSize;
    private int drainingStartQueueSize;

    private Promise<Void> promise = Promise.promise();
    private boolean readStreamEnded;
    private boolean handlingFinished;

    private AtomicInteger queueSize = new AtomicInteger();

    public ReadStreamConsumer(Function<T, Future<Void>> writeHandler) {
        this.writeHandler = writeHandler;
    }

    @Override
    public WriteStream<T> exceptionHandler(Handler<Throwable> exceptionHandler) {
        this.exceptionHandler = exceptionHandler;
        return this;
    }

    @Override
    public Future<Void> write(T data) {
        try {
            if (handlingFinished) {
                log.error("Handling already finished {}", this);
            }
            queueSize.incrementAndGet();
            return writeHandler.apply(data)
                    .map(r -> {
                        itemHandled(null);
                        return r;
                    })
                    .onFailure(this::handleException);
        } catch (Exception ex) {
            handleException(ex);
            return Future.failedFuture(ex);
        }
    }

    @Override
    public void write(T data, Handler<AsyncResult<Void>> handler) {
        try {
            if (handlingFinished) {
                log.error("Handling already finished {}", this);
            }
            queueSize.incrementAndGet();
            this.writeHandler.apply(data)
                    .onFailure(this::handleException)
                    .map(r -> {
                        itemHandled(null);
                        return r;
                    })
                    .onComplete(handler);
        } catch (Exception ex) {
            handleException(ex);
            handler.handle(promise.future());
        }
    }

    protected void itemHandled(Throwable ex) {
        int currentQueueSize = queueSize.decrementAndGet();
        if (drainingStartQueueSize > 0 && currentQueueSize <= drainingStartQueueSize) {
            drainingStartQueueSize = 0;
            if (drainHandler != null) {
                drainHandler.handle(null);
            }
        }
        checkEnd(ex);
    }

    protected void checkEnd(Throwable ex) {
        long currentSize = queueSize.get();
        if (ex != null) {
            promise.tryFail(ex);
            if (endHandler != null) {
                endHandler.handle(promise.future());
            }
        } else if (readStreamEnded && currentSize == 0) {
            handlingFinished = true;
            promise.tryComplete();
            if (endHandler != null) {
                endHandler.handle(promise.future());
            }
        }
    }

    protected void handleException(Throwable ex) {
        log.error("Handling failed", ex);
        itemHandled(ex);
        if (exceptionHandler != null) {
            exceptionHandler.handle(ex);
        }
    }

    @Override
    public void end(Handler<AsyncResult<Void>> handler) {
        endHandler = handler;
        readStreamEnded = true;
        checkEnd(null);
    }

    @Override
    public ReadStreamConsumer<T> setWriteQueueMaxSize(int maxSize) {
        this.maxQueueSize = maxSize;
        return this;
    }

    @Override
    public boolean writeQueueFull() {
        if (maxQueueSize > 0 && drainingStartQueueSize == 0 && queueSizeExceedsLimit()) {
            drainingStartQueueSize = Math.max(1, maxQueueSize / 2);
        }
        return drainingStartQueueSize > 0;
    }

    protected boolean queueSizeExceedsLimit() {
        return queueSize.get() >= maxQueueSize;
    }

    @Override
    public ReadStreamConsumer<T> drainHandler(Handler<Void> handler) {
        this.drainHandler = handler;
        return this;
    }

    public static <T> Future<Void> forEach(AsyncReadStream<T> readStream, Function<T, Future<Void>> handler) {
        return readStream.pipeTo(new ReadStreamConsumer<>(handler));
    }

    public static <T> Future<List<T>> toList(AsyncReadStream<T> readStream) {
        List<T> list = new ArrayList<>();
        return readStream.pipeTo(new ReadStreamConsumer<>(t -> {
            list.add(t);
            return Future.succeededFuture();
        })).map(list);
    }

}
