package org.jarbframework.constraint.database;

import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;
import org.jarbframework.utils.Asserts;
import org.jarbframework.utils.orm.ColumnReference;

/**
 * Describes that database constraints of a specific column.
 * 
 * @author Jeroen van Schagen
 * @since 20-05-2011
 */
public class ColumnMetadata {

    /** Reference to the column **/
    private final ColumnReference columnReference;

    /** Determines if a not-null value is required **/
    private boolean required;
    /** Determines if the database can auto increment **/
    private boolean autoIncrement;
    /** Describes the default value of our column **/
    private String defaultValue;
    /** Maximum length of the column **/
    private Integer maximumLength;
    /** Number radix, only used for numeric values **/
    private Integer radix;
    /** Number of fractional digits, only for numeric values **/
    private Integer fractionLength;

    /**
     * Construct a new {@link ColumnMetadata}.
     * @param columnReference reference to the column
     */
    public ColumnMetadata(ColumnReference columnReference) {
        this.columnReference = Asserts.notNull(columnReference, "Column reference cannot be null");
    }

    /**
     * Retrieve the column that we are describing.
     * @return column reference
     */
    public ColumnReference getColumnReference() {
        return columnReference;
    }

    public String getTableName() {
        return columnReference.getTableName();
    }

    public String getColumnName() {
        return columnReference.getColumnName();
    }

    /**
     * Determine if this column is required, whenever {@code true}
     * a not-null value is required.
     * @return {@code true} if required, else {@code false}
     */
    public boolean isRequired() {
        return required;
    }

    void setRequired(boolean required) {
        this.required = required;
    }

    /**
     * Determine if default column values can be generated by the database.
     * @return {@code true} if values can be generated, else {@code false}
     */
    public boolean isAutoIncrement() {
        return autoIncrement;
    }

    void setAutoIncrement(boolean autoIncrement) {
        this.autoIncrement = autoIncrement;
    }

    /**
     * Column values can be generated if the column has a default value it can use, or
     * an auto increment has been specified.
     * @param columnMetadata provides column information
     * @return {@code true} if the column value can be generated, else {@code false}
     */
    public boolean isGeneratable() {
        return hasDefaultValue() || isAutoIncrement();
    }

    /**
     * Determine if the column has a default value, used whenever no value
     * has been provided during the insert statement.
     * @return {@code true} if a default value is specified, else {@code false}
     */
    public boolean hasDefaultValue() {
        return defaultValue != null;
    }

    public Object getDefaultValue() {
        return defaultValue;
    }

    void setDefaultValue(String defaultValue) {
        this.defaultValue = defaultValue;
    }

    /**
     * Determine if our column has a maximum length. Whenever a column has no
     * maximum length, {@link #getMaximumLength()} always returns {@code null}.
     * @return {@code true} if the column has a maximum length, else {@code false}
     */
    public boolean hasMaximumLength() {
        return maximumLength != null;
    }

    /**
     * Retrieve the maximum length of a column, if any.
     * @return maximum length, if any
     */
    public Integer getMaximumLength() {
        return maximumLength;
    }

    void setMaximumLength(Integer maximumLength) {
        this.maximumLength = maximumLength;
    }

    /**
     * Retrieve the radix of a number.
     * @return number radix
     */
    public Integer getRadix() {
        return radix;
    }

    void setRadix(Integer radix) {
        this.radix = radix;
    }

    /**
     * Determine if our column has a maximum fraction length, meaning the number
     * of digits behind the comma. Fraction length restrictions are possible on
     * numeric column types, such as {@code double} or {@code numeric}. Whenever
     * a column has no maximum fraction length, {@link #getFractionLength()}
     * will always result in {@code null}.
     * @return {@code true} if the column has a maximum fraction length, else {@code false}
     */
    public boolean hasFractionLength() {
        return fractionLength != null;
    }

    /**
     * Retrieve the maximum fraction length, if any. Fraction length describes the
     * number of digits behind the comma. Fraction length restrictions are possible
     * on numeric column types, such as {@code double} or {@code numeric}. 
     * @return maximum fraction length, if any
     */
    public Integer getFractionLength() {
        return fractionLength;
    }

    void setFractionLength(Integer fractionLength) {
        this.fractionLength = fractionLength;
    }

    @Override
    public boolean equals(Object obj) {
        boolean equals = false;
        if (obj instanceof ColumnMetadata) {
            equals = columnReference.equals(((ColumnMetadata) obj).getColumnReference());
        }
        return equals;
    }

    @Override
    public int hashCode() {
        return columnReference.hashCode();
    }

    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this, ToStringStyle.SHORT_PREFIX_STYLE);
    }
}
