/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.clients;

import java.io.Closeable;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.apache.kafka.clients.MetadataCache;
import org.apache.kafka.common.Cluster;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.Node;
import org.apache.kafka.common.PartitionInfo;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.AuthenticationException;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.common.internals.ClusterResourceListeners;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.requests.MetadataRequest;
import org.apache.kafka.common.requests.MetadataResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Metadata
implements Closeable {
    private static final Logger log = LoggerFactory.getLogger(Metadata.class);
    public static final long TOPIC_EXPIRY_MS = 300000L;
    private static final long TOPIC_EXPIRY_NEEDS_UPDATE = -1L;
    private final long refreshBackoffMs;
    private final long metadataExpireMs;
    private int updateVersion;
    private int requestVersion;
    private long lastRefreshMs;
    private long lastSuccessfulRefreshMs;
    private AuthenticationException authenticationException;
    private MetadataCache cache = MetadataCache.empty();
    private boolean needUpdate;
    private final Map<String, Long> topics;
    private final List<Listener> listeners;
    private final ClusterResourceListeners clusterResourceListeners;
    private boolean needMetadataForAllTopics;
    private final boolean allowAutoTopicCreation;
    private final boolean topicExpiryEnabled;
    private boolean isClosed;
    private final Map<TopicPartition, Integer> lastSeenLeaderEpochs;

    public Metadata(long refreshBackoffMs, long metadataExpireMs, boolean allowAutoTopicCreation) {
        this(refreshBackoffMs, metadataExpireMs, allowAutoTopicCreation, false, new ClusterResourceListeners());
    }

    public Metadata(long refreshBackoffMs, long metadataExpireMs, boolean allowAutoTopicCreation, boolean topicExpiryEnabled, ClusterResourceListeners clusterResourceListeners) {
        this.refreshBackoffMs = refreshBackoffMs;
        this.metadataExpireMs = metadataExpireMs;
        this.allowAutoTopicCreation = allowAutoTopicCreation;
        this.topicExpiryEnabled = topicExpiryEnabled;
        this.lastRefreshMs = 0L;
        this.lastSuccessfulRefreshMs = 0L;
        this.requestVersion = 0;
        this.updateVersion = 0;
        this.needUpdate = false;
        this.topics = new HashMap<String, Long>();
        this.listeners = new ArrayList<Listener>();
        this.clusterResourceListeners = clusterResourceListeners;
        this.needMetadataForAllTopics = false;
        this.isClosed = false;
        this.lastSeenLeaderEpochs = new HashMap<TopicPartition, Integer>();
    }

    public synchronized Cluster fetch() {
        return this.cache.cluster();
    }

    public synchronized void add(String topic) {
        Objects.requireNonNull(topic, "topic cannot be null");
        if (this.topics.put(topic, -1L) == null) {
            this.requestUpdateForNewTopics();
        }
    }

    public synchronized long timeToAllowUpdate(long nowMs) {
        return Math.max(this.lastRefreshMs + this.refreshBackoffMs - nowMs, 0L);
    }

    public synchronized long timeToNextUpdate(long nowMs) {
        long timeToExpire = this.needUpdate ? 0L : Math.max(this.lastSuccessfulRefreshMs + this.metadataExpireMs - nowMs, 0L);
        return Math.max(timeToExpire, this.timeToAllowUpdate(nowMs));
    }

    public synchronized int requestUpdate() {
        this.needUpdate = true;
        return this.updateVersion;
    }

    public synchronized boolean updateLastSeenEpochIfNewer(TopicPartition topicPartition, int leaderEpoch) {
        Objects.requireNonNull(topicPartition, "TopicPartition cannot be null");
        return this.updateLastSeenEpoch(topicPartition, leaderEpoch, oldEpoch -> leaderEpoch > oldEpoch, true);
    }

    Optional<Integer> lastSeenLeaderEpoch(TopicPartition topicPartition) {
        return Optional.ofNullable(this.lastSeenLeaderEpochs.get(topicPartition));
    }

    private synchronized boolean updateLastSeenEpoch(TopicPartition topicPartition, int epoch, Predicate<Integer> epochTest, boolean setRequestUpdateFlag) {
        Integer oldEpoch = this.lastSeenLeaderEpochs.get(topicPartition);
        log.trace("Determining if we should replace existing epoch {} with new epoch {}", (Object)oldEpoch, (Object)epoch);
        if (oldEpoch == null || epochTest.test(oldEpoch)) {
            log.debug("Updating last seen epoch from {} to {} for partition {}", oldEpoch, epoch, topicPartition);
            this.lastSeenLeaderEpochs.put(topicPartition, epoch);
            if (setRequestUpdateFlag) {
                this.needUpdate = true;
            }
            return true;
        }
        log.debug("Not replacing existing epoch {} with new epoch {}", (Object)oldEpoch, (Object)epoch);
        return false;
    }

    public synchronized boolean updateRequested() {
        return this.needUpdate;
    }

    public synchronized Optional<PartitionInfo> partitionInfoIfCurrent(TopicPartition topicPartition) {
        Integer epoch = this.lastSeenLeaderEpochs.get(topicPartition);
        if (epoch == null) {
            return this.cache.getPartitionInfo(topicPartition);
        }
        return this.cache.getPartitionInfoHavingEpoch(topicPartition, epoch);
    }

    public synchronized AuthenticationException getAndClearAuthenticationException() {
        if (this.authenticationException != null) {
            AuthenticationException exception = this.authenticationException;
            this.authenticationException = null;
            return exception;
        }
        return null;
    }

    public synchronized void awaitUpdate(int lastVersion, long maxWaitMs) throws InterruptedException {
        if (maxWaitMs < 0L) {
            throw new IllegalArgumentException("Max time to wait for metadata updates should not be < 0 milliseconds");
        }
        long begin = System.currentTimeMillis();
        long remainingWaitMs = maxWaitMs;
        while (this.updateVersion <= lastVersion && !this.isClosed()) {
            long elapsed;
            AuthenticationException ex = this.getAndClearAuthenticationException();
            if (ex != null) {
                throw ex;
            }
            if (remainingWaitMs != 0L) {
                this.wait(remainingWaitMs);
            }
            if ((elapsed = System.currentTimeMillis() - begin) >= maxWaitMs) {
                throw new TimeoutException("Failed to update metadata after " + maxWaitMs + " ms.");
            }
            remainingWaitMs = maxWaitMs - elapsed;
        }
        if (this.isClosed()) {
            throw new KafkaException("Requested metadata update after close");
        }
    }

    public synchronized void setTopics(Collection<String> topics) {
        Set<TopicPartition> partitionsToRemove = this.lastSeenLeaderEpochs.keySet().stream().filter(tp -> !topics.contains(tp.topic())).collect(Collectors.toSet());
        partitionsToRemove.forEach(this.lastSeenLeaderEpochs::remove);
        this.cache.retainTopics(topics);
        if (!this.topics.keySet().containsAll(topics)) {
            this.requestUpdateForNewTopics();
        }
        this.topics.clear();
        for (String topic : topics) {
            this.topics.put(topic, -1L);
        }
    }

    public synchronized Set<String> topics() {
        return new HashSet<String>(this.topics.keySet());
    }

    public synchronized boolean containsTopic(String topic) {
        return this.topics.containsKey(topic);
    }

    public synchronized void bootstrap(List<InetSocketAddress> addresses, long now) {
        this.needUpdate = true;
        this.lastRefreshMs = now;
        this.lastSuccessfulRefreshMs = now;
        ++this.updateVersion;
        this.cache = MetadataCache.bootstrap(addresses);
    }

    public synchronized void update(MetadataResponse response, long now) {
        this.update(this.requestVersion, response, now);
    }

    public synchronized void update(int requestVersion, MetadataResponse metadataResponse, long now) {
        String newClusterId;
        Objects.requireNonNull(metadataResponse, "Metadata response cannot be null");
        if (this.isClosed()) {
            throw new IllegalStateException("Update requested after metadata close");
        }
        if (requestVersion == this.requestVersion) {
            this.needUpdate = false;
        } else {
            this.requestUpdate();
        }
        this.lastRefreshMs = now;
        this.lastSuccessfulRefreshMs = now;
        ++this.updateVersion;
        if (this.topicExpiryEnabled) {
            Iterator<Map.Entry<String, Long>> it = this.topics.entrySet().iterator();
            while (it.hasNext()) {
                Map.Entry<String, Long> entry = it.next();
                long expireMs = entry.getValue();
                if (expireMs == -1L) {
                    entry.setValue(now + 300000L);
                    continue;
                }
                if (expireMs > now) continue;
                it.remove();
                log.debug("Removing unused topic {} from the metadata list, expiryMs {} now {}", entry.getKey(), expireMs, now);
            }
        }
        String previousClusterId = this.cache.cluster().clusterResource().clusterId();
        this.cache = this.handleMetadataResponse(metadataResponse, topic -> true);
        Set<String> unavailableTopics = metadataResponse.unavailableTopics();
        Cluster clusterForListeners = this.cache.cluster();
        this.fireListeners(clusterForListeners, unavailableTopics);
        if (this.needMetadataForAllTopics) {
            this.needUpdate = false;
            this.cache = this.handleMetadataResponse(metadataResponse, this.topics.keySet()::contains);
        }
        if (!Objects.equals(previousClusterId, newClusterId = this.cache.cluster().clusterResource().clusterId())) {
            log.info("Cluster ID: {}", (Object)newClusterId);
        }
        this.clusterResourceListeners.onUpdate(clusterForListeners.clusterResource());
        this.notifyAll();
        log.debug("Updated cluster metadata version {} to {}", (Object)this.updateVersion, (Object)this.cache);
    }

    private MetadataCache handleMetadataResponse(MetadataResponse metadataResponse, Predicate<String> topicsToRetain) {
        HashSet<String> internalTopics = new HashSet<String>();
        ArrayList<MetadataCache.PartitionInfoAndEpoch> partitions = new ArrayList<MetadataCache.PartitionInfoAndEpoch>();
        for (MetadataResponse.TopicMetadata metadata : metadataResponse.topicMetadata()) {
            if (!topicsToRetain.test(metadata.topic()) || metadata.error() != Errors.NONE) continue;
            if (metadata.isInternal()) {
                internalTopics.add(metadata.topic());
            }
            for (MetadataResponse.PartitionMetadata partitionMetadata : metadata.partitionMetadata()) {
                this.updatePartitionInfo(metadata.topic(), partitionMetadata, partitionInfo -> {
                    int epoch = partitionMetadata.leaderEpoch().orElse(-1);
                    partitions.add(new MetadataCache.PartitionInfoAndEpoch((PartitionInfo)partitionInfo, epoch));
                });
            }
        }
        return new MetadataCache(metadataResponse.clusterId(), new ArrayList<Node>(metadataResponse.brokers()), partitions, metadataResponse.topicsByError(Errors.TOPIC_AUTHORIZATION_FAILED), metadataResponse.topicsByError(Errors.INVALID_TOPIC_EXCEPTION), internalTopics, metadataResponse.controller());
    }

    private void updatePartitionInfo(String topic, MetadataResponse.PartitionMetadata partitionMetadata, Consumer<PartitionInfo> partitionInfoConsumer) {
        TopicPartition tp = new TopicPartition(topic, partitionMetadata.partition());
        if (partitionMetadata.leaderEpoch().isPresent()) {
            int newEpoch = partitionMetadata.leaderEpoch().get();
            if (this.updateLastSeenEpoch(tp, newEpoch, oldEpoch -> newEpoch >= oldEpoch, false)) {
                partitionInfoConsumer.accept(MetadataResponse.partitionMetaToInfo(topic, partitionMetadata));
            } else {
                PartitionInfo previousInfo = this.cache.cluster().partition(tp);
                if (previousInfo != null) {
                    partitionInfoConsumer.accept(previousInfo);
                } else if (this.containsTopic(topic)) {
                    log.debug("Got an older epoch in partition metadata response for {}, but we are not tracking this topic. Ignoring metadata update for this partition", (Object)tp);
                } else {
                    log.warn("Got an older epoch in partition metadata response for {}, but could not find previous partition info to use. Refusing to update metadata for this partition", (Object)tp);
                }
            }
        } else {
            this.lastSeenLeaderEpochs.clear();
            partitionInfoConsumer.accept(MetadataResponse.partitionMetaToInfo(topic, partitionMetadata));
        }
    }

    private void fireListeners(Cluster newCluster, Set<String> unavailableTopics) {
        for (Listener listener : this.listeners) {
            listener.onMetadataUpdate(newCluster, unavailableTopics);
        }
    }

    public synchronized void failedUpdate(long now, AuthenticationException authenticationException) {
        this.lastRefreshMs = now;
        this.authenticationException = authenticationException;
        if (authenticationException != null) {
            this.notifyAll();
        }
    }

    public synchronized int updateVersion() {
        return this.updateVersion;
    }

    public synchronized long lastSuccessfulUpdate() {
        return this.lastSuccessfulRefreshMs;
    }

    public boolean allowAutoTopicCreation() {
        return this.allowAutoTopicCreation;
    }

    public synchronized void needMetadataForAllTopics(boolean needMetadataForAllTopics) {
        if (needMetadataForAllTopics && !this.needMetadataForAllTopics) {
            this.requestUpdateForNewTopics();
        }
        this.needMetadataForAllTopics = needMetadataForAllTopics;
    }

    public synchronized boolean needMetadataForAllTopics() {
        return this.needMetadataForAllTopics;
    }

    public synchronized void addListener(Listener listener) {
        this.listeners.add(listener);
    }

    public synchronized void removeListener(Listener listener) {
        this.listeners.remove(listener);
    }

    @Override
    public synchronized void close() {
        this.isClosed = true;
        this.notifyAll();
    }

    public synchronized boolean isClosed() {
        return this.isClosed;
    }

    synchronized void requestUpdateForNewTopics() {
        this.lastRefreshMs = 0L;
        ++this.requestVersion;
        this.requestUpdate();
    }

    public synchronized MetadataRequestAndVersion newMetadataRequestAndVersion() {
        MetadataRequest.Builder metadataRequestBuilder = this.needMetadataForAllTopics ? MetadataRequest.Builder.allTopics() : new MetadataRequest.Builder(new ArrayList<String>(this.topics.keySet()), this.allowAutoTopicCreation());
        return new MetadataRequestAndVersion(metadataRequestBuilder, this.requestVersion);
    }

    public static class MetadataRequestAndVersion {
        public final MetadataRequest.Builder requestBuilder;
        public final int requestVersion;

        private MetadataRequestAndVersion(MetadataRequest.Builder requestBuilder, int requestVersion) {
            this.requestBuilder = requestBuilder;
            this.requestVersion = requestVersion;
        }
    }

    public static interface Listener {
        public void onMetadataUpdate(Cluster var1, Set<String> var2);
    }
}

