/*
 *  jDTAUS - DTAUS fileformat.
 *  Copyright (c) 2005 Christian Schulte <cs@schulte.it>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */
package org.jdtaus.core.container;

/**
 * Implementation container.
 * <p>The container is responsible for creating and maintaining instances of
 * implementations of specifications. It is driven by a model defining the
 * relationships between specifications and implementations. Its main use is to
 * decouple compiletime classpaths from runtime classpaths. Implementations need
 * not be visible during compiletime. Applications can be written against a
 * stable API with no implementation dependent knowledge. The implementations
 * building the runtime can then be exchanged independently during application
 * assembly.</p>
 * <p>Example: Getting an instance of an implementation of a specification<br/>
 * <pre>
 * Specification spec = (Specification) ContainerFactory.getContainer().
 *     getImplementation(Specification.class, "jDTAUS Implementation");
 * </pre></p>
 * <p>Example: Getting an instance of an implementation of a dependency<br/>
 * <pre>
 * public class Implementation {
 *
 *   private transient Specification _dependency0;
 *
 *   private Specification getSpecification() {
 *       Specification ret = null;
 *       if(this._dependency0 != null) {
 *           ret = this._dependency0;
 *       } else {
 *           ret = (Specification) ContainerFactory.getContainer().
 *               getDependency(Implementation.class, "Specification");
 *
 *           if(ModelFactory.getModel().getModules().
 *               getImplementation(Implementation.class.getName()).
 *               getDependencies().getDependency("Specification").isBound()) {
 *
 *               this._dependency0 = ret;
 *           }
 *       }
 *
 *       return ret;
 *   }
 *
 * }
 * </pre></p>
 *
 * @author <a href="mailto:cs@schulte.it">Christian Schulte</a>
 * @version $Id: Container.java 1914 2007-03-01 02:20:44Z schulte2005 $
 *
 * @see ContainerFactory
 */
public interface Container
{

    //--Container---------------------------------------------------------------

    /**
     * Gets an instance of an implementation of some specification.
     *
     * @param specification the class of the specification to return an
     * instance of an implementation for.
     * @param implementationName the name of the implementation to return.
     *
     * @return an instance of the implementation named
     * {@code implementationName} implementing {@code specification}.
     *
     * @throws NullPointerException if either {@code specification} or
     * {@code implementationName} is {@code null}.
     * @throws org.jdtaus.core.container.InstantiationException if creating
     * an instance of the implementation fails.
     */
    Object getImplementation(Class specification, String implementationName);

    /**
     * Gets an instance of an implementation of some dependency.
     *
     * @param implementation the class of the implementation to return an
     * instance of a dependency for.
     * @param dependencyName the logical name of the dependency to return.
     *
     * @return an instance of the implementation of the dependency named
     * {@code dependencyName} for {@code implementation}.
     *
     * @throws NullPointerException if either {@code implementation} or
     * {@code dependencyName} is {@code null}.
     * @throws org.jdtaus.core.container.InstantiationException if creating
     * an instance of the dependency fails.
     */
    Object getDependency(Class implementation, String dependencyName);

    //---------------------------------------------------------------Container--

}
