/*
 *  jDTAUS - DTAUS fileformat.
 *  Copyright (c) 2005 Christian Schulte <cs@schulte.it>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */
package org.jdtaus.core.container;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

/**
 * Factory for the {@code Container} singleton.
 *
 * @author <a href="mailto:cs@schulte.it">Christian Schulte</a>
 * @version $Id: ContainerFactory.java 3212 2007-04-29 22:51:15Z schulte2005 $
 */
public abstract class ContainerFactory
{

    //--Constants---------------------------------------------------------------

    /** Default {@code Container} implementation. */
    private static final String DEFAULT_CONTAINER =
        "org.jdtaus.core.container.ri.client.DefaultContainer";

    /** Empty array. */
    static final Class[] EMPTY = {};

    //---------------------------------------------------------------Constants--
    //--ContainerFactory--------------------------------------------------------

    /** Default singleton instance. */
    private static Container instance;

    /**
     * Gets the {@code Container} singleton.
     * <p>By default this class will instantiate a new container and hold it in
     * a static class variable as the singleton to return for other calls. This
     * behaviour can be changed by setting a system property with key
     * {@code org.jdtaus.core.container.ContainerFactory} to the name of a
     * class defining a {@code public static Container getContainer()} method
     * returning the singleton instance of {@code Container}.</p>
     *
     * @return the singleton {@code Container} instance.
     *
     * @throws ContainerError for unrecoverable container errors.
     *
     * @see ContainerFactory#newContainer()
     */
    public static Container getContainer()
    {
        Object ret = null;
        final String factory =
            System.getProperty(ContainerFactory.class.getName());

        try
        {
            if(factory != null)
            {
                final Class clazz;
                final Method meth;

                // Call getContainer() on that class.
                clazz = ContainerFactory.loadClass(factory);
                meth = clazz.getDeclaredMethod("getContainer",
                    ContainerFactory.EMPTY);

                ret = meth.invoke(null, (Object[]) ContainerFactory.EMPTY);
            }
            else
            {
                if(ContainerFactory.instance == null)
                {
                    ContainerFactory.instance =
                        ContainerFactory.newContainer();

                }

                ret = ContainerFactory.instance;
            }

            return (Container) ret;
        }
        catch (SecurityException e)
        {
            throw new ContainerError(e);
        }
        catch (NoSuchMethodException e)
        {
            throw new ContainerError(e);
        }
        catch (IllegalAccessException e)
        {
            throw new ContainerError(e);
        }
        catch (InvocationTargetException e)
        {
            throw new ContainerError(e.getTargetException() == null ?
                e : e.getTargetException());

        }
        catch(ClassCastException e)
        {
            throw new ContainerError(e);
        }
    }

    /**
     * Creates a new instance of the {@code Container} singleton implementation.
     * <p>The container implementation to be used can be controlled via a system
     * property with key {@code org.jdtaus.core.container.Container} set to a
     * class name to be loaded as the container implementation.</p>
     * <p>This method should be used by {@code getContainer()} implementors to
     * retrieve a new {@code Container} instance.</p>
     *
     * @return a new instance of the configured {@code Container}
     * implementation.
     *
     * @throws ContainerError for unrecoverable container errors.
     */
    public static Container newContainer()
    {
        Object instance = null;
        final String impl = System.getProperty(Container.class.getName(),
            ContainerFactory.DEFAULT_CONTAINER);

        try
        {
            final Constructor ctor;
            final Class clazz = ContainerFactory.loadClass(impl);

            ctor = clazz.getDeclaredConstructor(ContainerFactory.EMPTY);
            ctor.setAccessible(true);
            instance = ctor.newInstance((Object[]) ContainerFactory.EMPTY);

            return (Container) instance;
        }
        catch (SecurityException e)
        {
            throw new ContainerError(e);
        }
        catch (NoSuchMethodException e)
        {
            throw new ContainerError(e);
        }
        catch (IllegalAccessException e)
        {
            throw new ContainerError(e);
        }
        catch (java.lang.InstantiationException e)
        {
            throw new ContainerError(e);
        }
        catch (InvocationTargetException e)
        {
            throw new ContainerError(e.getTargetException() == null ?
                e : e.getTargetException());

        }
        catch(ClassCastException e)
        {
            throw new ContainerError(e);
        }
    }

    static ClassLoader getClassLoader()
    {
        ClassLoader classLoader = Thread.currentThread().
            getContextClassLoader();

        if(classLoader == null)
        {
            classLoader = ClassLoader.getSystemClassLoader();
        }

        assert classLoader != null :
            "Expected system classloader to always be available.";

        return classLoader;
    }

    static Class loadClass(final String name)
    {
        try
        {
            return ContainerFactory.getClassLoader().loadClass(name);
        }
        catch (ClassNotFoundException e)
        {
            throw new ContainerError(e);
        }
    }

    //--------------------------------------------------------ContainerFactory--

}
