/*
 *  jDTAUS - DTAUS fileformat.
 *  Copyright (c) 2005 Christian Schulte <cs@schulte.it>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */
package org.jdtaus.core.container;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

/**
 * Factory for the {@code Context} singleton.
 *
 * @author <a href="mailto:cs@schulte.it">Christian Schulte</a>
 * @version $Id: ContextFactory.java 2756 2007-04-14 02:01:15Z schulte2005 $
 */
public abstract class ContextFactory
{

    //--Constants---------------------------------------------------------------

    /** Default {@code Context} implementation. */
    private static final String DEFAULT_CONTEXT =
        "org.jdtaus.core.container.ri.client.DefaultContext";

    //---------------------------------------------------------------Constants--
    //--ContextFactory----------------------------------------------------------

    /** Default singleton instance. */
    private static Context instance;

    /**
     * Gets the {@code Context} singleton.
     * <p>By default this class will instantiate a new context and hold it in a
     * static class variable as the singleton to return for other calls. This
     * behaviour can be changed by setting a system property with key
     * {@code org.jdtaus.core.container.ContextFactory} to the name of a
     * class defining a {@code public static Context getContext()} method
     * returning the singleton instance of {@code Context}.</p>
     *
     * @return the singleton {@code Context} instance.
     *
     * @throws ContextError for unrecoverable context errors.
     *
     * @see ContextFactory#newContext()
     */
    public static Context getContext()
    {
        Object ret = null;
        final String factory =
            System.getProperty(ContextFactory.class.getName());

        try
        {
            if(factory != null)
            {
                final Class clazz;
                final Method meth;

                // Call getContext() on that class.
                clazz = ContainerFactory.loadClass(factory);
                meth = clazz.getDeclaredMethod("getContext",
                    ContainerFactory.EMPTY);

                ret = meth.invoke(null, (Object[]) ContainerFactory.EMPTY);
            }
            else
            {
                if(ContextFactory.instance == null)
                {
                    ContextFactory.instance =
                        ContextFactory.newContext();

                }

                ret = ContextFactory.instance;
            }

            return (Context) ret;
        }
        catch (SecurityException e)
        {
            throw new ContextError(e);
        }
        catch (NoSuchMethodException e)
        {
            throw new ContextError(e);
        }
        catch (IllegalAccessException e)
        {
            throw new ContextError(e);
        }
        catch (InvocationTargetException e)
        {
            throw new ContextError(e.getTargetException() == null ?
                e : e.getTargetException());

        }
        catch(ClassCastException e)
        {
            throw new ContextError(e);
        }
    }

    /**
     * Creates a new instance of the configured {@code Context} implementation.
     * <p>The context implementation to be used can be controlled via a system
     * property with key {@code org.jdtaus.core.container.Context} set
     * to a class name to be loaded as the context implementation.</p>
     * <p>This method should be used by {@code getContext()} implementors to
     * retrieve a new {@code Context} instance.</p>
     *
     * @return a new instance of the configured {@code Context}
     * implementation.
     *
     * @throws ContextError for unrecoverable context errors.
     */
    public static Context newContext()
    {
        Object instance = null;
        String impl = System.getProperty(Context.class.getName(),
            ContextFactory.DEFAULT_CONTEXT);

        try
        {
            final Constructor ctor;
            final Class clazz = ContainerFactory.loadClass(impl);

            ctor = clazz.getDeclaredConstructor(ContainerFactory.EMPTY);
            ctor.setAccessible(true);
            instance = ctor.newInstance((Object[]) ContainerFactory.EMPTY);

            return (Context) instance;
        }
        catch (SecurityException e)
        {
            throw new ContextError(e);
        }
        catch (NoSuchMethodException e)
        {
            throw new ContextError(e);
        }
        catch (IllegalAccessException e)
        {
            throw new ContextError(e);
        }
        catch (java.lang.InstantiationException e)
        {
            throw new ContextError(e);
        }
        catch (InvocationTargetException e)
        {
            throw new ContextError(e.getTargetException() == null ?
                e : e.getTargetException());

        }
        catch(ClassCastException e)
        {
            throw new ContextError(e);
        }
    }

    //----------------------------------------------------------ContextFactory--

}
