/*
 *  jDTAUS - DTAUS fileformat.
 *  Copyright (c) 2005 Christian Schulte <cs@schulte.it>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */
package org.jdtaus.core.container;

import java.io.Serializable;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

/**
 * Dependency meta-data.
 * <p>A dependency consists of a name uniquely identifying the dependency in a
 * collection of dependencies. A dependency pairs a specification with a
 * corresponding implementation from a set of available implementations.
 * Properties set with a dependency overwrite the properties defined by the
 * implementation. The {@code bound} flag indicates if an instance of the
 * dependency is bound to the requesting implementation.</p>
 *
 * @author <a href="mailto:cs@schulte.it">Christian Schulte</a>
 * @version $Id: Dependency.java 2230 2007-03-26 01:37:48Z schulte2005 $
 */
public class Dependency implements Cloneable, Serializable
{

    //--Dependency--------------------------------------------------------------

    /**
     * The name of the dependency.
     * @serial
     */
    private String name;

    /**
     * The specification of the dependency.
     * @serial
     */
    private Specification specification;

    /**
     * The implementation of the dependency.
     * @serial
     */
    private Implementation implementation;

    /**
     * The properties of the dependency.
     * @serial
     */
    private Properties properties;
    private transient Properties cachedProperties;

    /**
     * Flag indicating if the dependency is bound to the requesting
     * implementation.
     * @serial
     */
    private boolean bound;

    /**
     * Gets the name of the dependency.
     *
     * @return the name of the dependency.
     */
    public String getName()
    {
        if(this.name == null)
        {
            this.name = "";
        }

        return name;
    }

    /**
     * Setter for property {@code name}.
     *
     * @param value the new name of the dependency.
     */
    public void setName(final String value)
    {
        this.name = value;
    }

    /**
     * Gets the specification of the dependency.
     *
     * @return the specification of the dependency.
     */
    public Specification getSpecification()
    {
        return this.specification;
    }

    /**
     * Setter for property {@code specification}.
     *
     * @param value the new specification of the dependency.
     */
    public void setSpecification(final Specification value)
    {
        this.specification = value;
    }

    /**
     * Gets the implementation of the dependency.
     *
     * @return the implementation of the dependency.
     */
    public Implementation getImplementation()
    {
        return this.implementation;
    }

    /**
     * Setter for property {@code implementation}.
     *
     * @param value the new implementation of the dependency.
     */
    public void setImplementation(final Implementation value)
    {
        this.implementation = value;
        this.cachedProperties = null;
    }

    /**
     * Gets the properties of the dependency.
     *
     * @return the properties of the dependency.
     */
    public Properties getProperties()
    {
        if(this.cachedProperties == null)
        {
            if(this.properties == null)
            {
                this.properties = new Properties();
            }

            Property iProp;
            Properties iProps = this.getImplementation().getProperties();
            final Set props = new HashSet();
            this.cachedProperties = new Properties();

            for(int i = iProps.size() - 1; i >= 0; i--)
            {
                iProp = iProps.getProperty(i);

                try
                {
                    this.properties.getProperty(iProp.getName());
                }
                catch(MissingPropertyException e)
                {
                    props.add(iProp);
                }
            }

            props.addAll(Arrays.asList(this.properties.getProperties()));
            this.cachedProperties.setProperties((Property[]) props.
                toArray(new Property[props.size()]));

        }

        return this.cachedProperties;
    }

    /**
     * Setter for property {@code properties}.
     *
     * @param value the new properties of the dependency.
     */
    public void setProperties(final Properties value)
    {
        this.properties = value;
        this.cachedProperties = null;
    }

    /**
     * Gets the flag indicating if the dependency is bound to a requesting
     * implementation.
     *
     * @return {@code true} if an instance of the dependency is bound to the
     * requesting implementation; {@code false} if not.
     */
    public boolean isBound()
    {
        return this.bound;
    }

    /**
     * Setter for property {@code bound}.
     *
     * @param value {@code true} if an instance of the dependency should be
     * bound to the requesting implementation; {@code false} if not.
     */
    public void setBound(boolean value)
    {
        this.bound = value;
    }

    /**
     * Creates a string representing the properties of the instance.
     *
     * @return a string representing the properties of the instance.
     */
    private String internalString()
    {
        return new StringBuffer(500).
            append("\n\tbound=").append(this.bound).
            append("\n\timplementation=").append(this.implementation).
            append("\n\tname=").append(this.name).
            append("\n\tproperties=").append(this.properties).
            append("\n\tspecification=").append(this.specification).
            toString();

    }

    //--------------------------------------------------------------Dependency--
    //--Object------------------------------------------------------------------

    /**
     * Returns a string representation of the object.
     *
     * @return a string representation of the object.
     */
    public String toString()
    {
        return super.toString() + this.internalString();
    }

    /**
     * Indicates whether some other object is equal to this one by comparing
     * the values of all properties.
     *
     * @param o the reference object with which to compare.
     *
     * @return {@code true} if this object is the same as {@code o};
     * {@code false} otherwise.
     */
    public boolean equals(final Object o)
    {
        boolean equal = this == o;

        if(!equal && o instanceof Dependency)
        {
            final Dependency that = (Dependency) o;
            equal = this.getName().equals(that.getName()) &&
                (this.implementation == null ? that.implementation == null :
                    this.implementation.equals(that.implementation)) &&
                (this.specification == null ? that.specification == null :
                    this.specification.equals(that.specification));

        }

        return equal;
    }

    /**
     * Returns a hash code value for this object.
     *
     * @return a hash code value for this object.
     */
    public int hashCode()
    {
        return this.getName().hashCode() +
            (this.getImplementation() == null ?
                0 : this.getImplementation().hashCode()) +
            (this.getSpecification() == null ?
                0 : this.getSpecification().hashCode());

    }

    /**
     * Creates and returns a deep copy of this object.
     *
     * @return a clone of this instance.
     */
    public Object clone()
    {
        try
        {
            final Dependency ret = (Dependency) super.clone();
            if(this.implementation != null)
            {
                ret.implementation =
                    (Implementation) this.implementation.clone();

            }
            if(this.properties != null)
            {
                ret.properties = (Properties) this.properties.clone();
            }
            if(this.specification != null)
            {
                ret.specification = (Specification) this.specification.clone();
            }

            return ret;
        }
        catch(CloneNotSupportedException e)
        {
            throw new AssertionError(e);
        }
    }

    //------------------------------------------------------------------Object--
}
