/*
 *  jDTAUS - DTAUS fileformat.
 *  Copyright (c) 2005 Christian Schulte <cs@schulte.it>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */
package org.jdtaus.core.container;

import java.io.Serializable;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

/**
 * Implementation meta-data.
 * <p>An implementation consists of the properties {@code identifier},
 * {@code name}, {@code vendor} and {@code version}. Property
 * {@code identifier} holds an identifier uniquely identifying the
 * implementation in a collection of implementations. Property {@code name}
 * holds a name of the implementation uniquely identifying the implementation
 * for a specification. Property {@code vendor} holds vendor information for the
 * vendor providing the implementation. Property {@code version} holds a textual
 * version of the implementation. Properties, dependencies and implemented
 * specifications may be inherited from a {@code parent} up the hierarchy.</p>
 *
 * @author <a href="mailto:cs@schulte.it">Christian Schulte</a>
 * @version $Id: Implementation.java 2230 2007-03-26 01:37:48Z schulte2005 $
 */
public class Implementation implements Cloneable, Serializable
{

    //--Implementation----------------------------------------------------------

    /**
     * The parent to inherit from.
     * @serial
     */
    private Implementation parent;

    /**
     * The name of the module holding the implementation.
     * @serial
     */
    private String moduleName;

    /**
     * The specifications the implementation implements.
     * @serial
     */
    private Specifications implementedSpecifications;
    private transient Specifications cachedSpecifications;

    /**
     * The dependencies of the implementation.
     * @serial
     */
    private Dependencies dependencies;
    private transient Dependencies cachedDependencies;

    /**
     * The properties of the implementation.
     * @serial
     */
    private Properties properties;
    private transient Properties cachedProperties;

    /**
     * The identifier of the implementation.
     * @serial
     */
    private String identifier;

    /**
     * The name of the implementation.
     * @serial
     */
    private String name;

    /**
     * The vendor of the implementation.
     * @serial
     */
    private String vendor;

    /**
     * The version of the implementation.
     * @serial
     */
    private String version;

    /**
     * Gets the name of the module holding the implementation.
     *
     * @return the name of the module holding the implementation.
     */
    public String getModuleName()
    {
        if(this.moduleName == null)
        {
            this.moduleName = "";
        }

        return this.moduleName;
    }

    /**
     * Setter for property {@code moduleName}.
     *
     * @param value the new name of the module holding the implementation.
     */
    public void setModuleName(final String value)
    {
        this.moduleName = value;
    }

    /**
     * Gets the specifications the implementation implements.
     *
     * @return the specifications the implementation implements.
     */
    public Specifications getImplementedSpecifications()
    {
        if(this.cachedSpecifications == null)
        {
            if(this.implementedSpecifications == null)
            {
                this.implementedSpecifications = new Specifications();
            }

            this.cachedSpecifications = this.implementedSpecifications;

            if(this.getParent() != null)
            {
                Specification pSpec;
                final Specifications pSpecs = this.getParent().
                    getImplementedSpecifications();

                final Set specs = new HashSet();

                for(int i = pSpecs.size() - 1; i >= 0; i--)
                {
                    pSpec = pSpecs.getSpecification(i);

                    try
                    {
                        this.implementedSpecifications.
                            getSpecification(pSpec.getIdentifier());

                    }
                    catch(MissingSpecificationException e)
                    {
                        specs.add(pSpec);
                    }
                }

                specs.addAll(Arrays.asList(
                    this.implementedSpecifications.getSpecifications()));

                this.cachedSpecifications = new Specifications();
                this.cachedSpecifications.setSpecifications((Specification[])
                specs.toArray(new Specification[specs.size()]));

            }
        }

        return this.cachedSpecifications;
    }

    /**
     * Setter for property {@code implementedSpecifications}.
     *
     * @param value the new specifications the implementation implements.
     */
    public void setImplementedSpecifications(final Specifications value)
    {
        this.implementedSpecifications = value;
        this.cachedSpecifications = null;
    }

    /**
     * Gets the dependencies of the implementation.
     *
     * @return the dependencies the implementation depends on.
     */
    public Dependencies getDependencies()
    {
        if(this.cachedDependencies == null)
        {
            if(this.dependencies == null)
            {
                this.dependencies = new Dependencies();
            }

            this.cachedDependencies = this.dependencies;

            if(this.getParent() != null)
            {
                Dependency pDep;
                final Dependencies pDeps = this.getParent().getDependencies();
                final Set deps = new HashSet();

                for(int i = pDeps.size() - 1; i >= 0; i--)
                {
                    pDep = pDeps.getDependency(i);

                    try
                    {
                        this.dependencies.getDependency(pDep.getName());
                    }
                    catch(MissingDependencyException e)
                    {
                        deps.add(pDep);
                    }
                }

                deps.addAll(Arrays.asList(
                    this.dependencies.getDependencies()));

                this.cachedDependencies = new Dependencies();
                this.cachedDependencies.setDependencies((Dependency[]) deps.
                    toArray(new Dependency[deps.size()]));

            }
        }

        return this.cachedDependencies;
    }

    /**
     * Setter for property {@code dependencies}.
     *
     * @param value the new dependencies of the implementation.
     */
    public void setDependencies(final Dependencies value)
    {
        this.dependencies = value;
        this.cachedDependencies = null;
    }

    /**
     * Gets the properties of the implementation.
     *
     * @return the properties of the implementation.
     */
    public Properties getProperties()
    {
        if(this.cachedProperties == null)
        {
            if(this.properties == null)
            {
                this.properties = new Properties();
            }

            this.cachedProperties = this.properties;

            if(this.getParent() != null)
            {
                Property pProp;
                final Properties pProps = this.getParent().getProperties();
                final Set props = new HashSet();
                for(int i = pProps.size() - 1; i >= 0; i--)
                {
                    pProp = pProps.getProperty(i);

                    try
                    {
                        this.properties.getProperty(pProp.getName());
                    }
                    catch(MissingPropertyException e)
                    {
                        props.add(pProp);
                    }
                }

                props.addAll(Arrays.asList(
                    this.properties.getProperties()));

                this.cachedProperties = new Properties();
                this.cachedProperties.setProperties((Property[]) props.
                    toArray(new Property[props.size()]));

            }
        }

        return this.cachedProperties;
    }

    /**
     * Setter for property {@code properties}.
     *
     * @param value new properties of the implementation.
     */
    public void setProperties(final Properties value)
    {
        this.properties = value;
        this.cachedProperties = null;
    }

    /**
     * Gets the identifier of the implementation.
     *
     * @return the unique identifier of the implementation.
     */
    public String getIdentifier()
    {
        if(this.identifier == null)
        {
            this.identifier = "";
        }

        return this.identifier;
    }

    /**
     * Setter for property {@code identifier}.
     *
     * @param value the new identifier of the implementation.
     */
    public void setIdentifier(final String value)
    {
        this.identifier = value;
    }

    /**
     * Gets the name of the implementation.
     *
     * @return the name of the implementation.
     */
    public String getName()
    {
        if(this.name == null)
        {
            this.name = "";
        }

        return name;
    }

    /**
     * Setter for property {@code name}.
     *
     * @param value the new name of the implementation.
     */
    public void setName(final String value)
    {
        this.name = value;
    }

    /**
     * Gets the parent implementation the implementation inherits from.
     *
     * @return the parent implementation the implementation inherits from or
     * {@code null} if the implementation has no parent.
     */
    public Implementation getParent()
    {
        return parent;
    }

    /**
     * Setter for property {@code parent}.
     *
     * @param value the new parent implementation of the implementation.
     */
    public void setParent(final Implementation value)
    {
        this.parent = value;
        this.cachedDependencies = null;
        this.cachedProperties = null;
        this.cachedSpecifications = null;
    }

    /**
     * Gets the vendor of the implementation.
     *
     * @return the vendor of the implementation.
     */
    public String getVendor()
    {
        if(this.vendor == null)
        {
            this.vendor = "";
        }

        return this.vendor;
    }

    /**
     * Setter for property {@code name}.
     *
     * @param value the new vendor of the implementation.
     */
    public void setVendor(final String value)
    {
        this.vendor = value;
    }

    /**
     * Gets the version of the implementation.
     *
     * @return the version of the implementation.
     */
    public String getVersion()
    {
        if(this.version == null)
        {
            this.version = "";
        }

        return this.version;
    }

    /**
     * Setter for property {@code version}.
     *
     * @param value the new version of the implementation.
     */
    public void setVersion(final String value)
    {
        this.version = value;
    }

    /**
     * Creates a string representing the properties of the instance.
     *
     * @return a string representing the properties of the instance.
     */
    private String internalString()
    {
        return new StringBuffer(500).
            append("\n\tdependencies=").append(this.dependencies).
            append("\n\tidentifier=").append(this.identifier).
            append("\n\timplementedSpecifications=").
            append(this.implementedSpecifications).
            append("\n\tmoduleName=").append(this.moduleName).
            append("\n\tname=").append(this.name).
            append("\n\tparent=").append(this.parent).
            append("\n\tproperties=").append(this.properties).
            append("\n\tvendor=").append(this.vendor).
            append("\n\tversion=").append(this.version).
            toString();

    }

    //----------------------------------------------------------Implementation--
    //--Object------------------------------------------------------------------

    /**
     * Returns a string representation of the object.
     *
     * @return a string representation of the object.
     */
    public String toString()
    {
        return super.toString() + this.internalString();
    }

    /**
     * Creates and returns a deep copy of this object.
     *
     * @return a clone of this instance.
     */
    public Object clone()
    {
        try
        {
            final Implementation ret = (Implementation) super.clone();
            if(this.parent != null)
            {
                ret.parent = (Implementation) this.parent.clone();
            }
            if(this.implementedSpecifications != null)
            {
                ret.implementedSpecifications =
                    (Specifications) this.implementedSpecifications.clone();

            }
            if(this.properties != null)
            {
                ret.properties = (Properties) this.properties.clone();
            }
            if(this.dependencies != null)
            {
                ret.dependencies = (Dependencies) this.dependencies.clone();
            }

            return ret;
        }
        catch(CloneNotSupportedException e)
        {
            throw new AssertionError(e);
        }
    }

    /**
     * Indicates whether some other object is equal to this one by comparing
     * property {@code identifier}.
     *
     * @param o the reference object with which to compare.
     *
     * @return {@code true} if this object is the same as {@code o};
     * {@code false} otherwise.
     */
    public final boolean equals(final Object o)
    {
        return o == this || (o != null && o instanceof Implementation &&
            ((Implementation)o).getIdentifier().equals(this.getIdentifier()));

    }

    /**
     * Returns a hash code value for this object.
     *
     * @return a hash code value for this object.
     */
    public final int hashCode()
    {
        return this.getIdentifier().hashCode();
    }

    //------------------------------------------------------------------Object--

}
