/*
 *  jDTAUS - DTAUS fileformat.
 *  Copyright (c) 2005 Christian Schulte <cs@schulte.it>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */
package org.jdtaus.core.container;

import java.io.Serializable;

/**
 * Module meta-data.
 * <p>A module consists of the properties {@code name}, {@code description}
 * and {@code version}. Property {@code name} holds the name of the module
 * uniquely identifying the module in a collection of modules. Property
 * {@code description} holds a textual description, property {@code version}
 * a textual version of the module. A module defines specifications and
 * implementations. Properties can be additionally defined.</p>
 *
 * @author <a href="mailto:cs@schulte.it">Christian Schulte</a>
 * @version $Id: Module.java 2201 2007-03-21 23:59:00Z schulte2005 $
 */
public class Module implements Cloneable, Serializable
{

    //--Module------------------------------------------------------------------

    /**
     * The specifications of the module.
     * @serial
     */
    private Specifications specifications;

    /**
     * The implementations of the module.
     * @serial
     */
    private Implementations implementations;

    /**
     * The properties of the module.
     * @serial
     */
    private Properties properties;

    /**
     * The description of the module.
     * @serial
     */
    private String description;

    /**
     * The name of the module.
     * @serial
     */
    private String name;

    /**
     * The version of the module.
     * @serial
     */
    private String version;

    /**
     * Gets the specifications of the module.
     *
     * @return the specifications of the module.
     */
    public Specifications getSpecifications()
    {
        if(this.specifications == null)
        {
            this.specifications = new Specifications();
        }

        return this.specifications;
    }

    /**
     * Setter for property {@code specifications}.
     *
     * @param value the new specifications of the module.
     */
    public void setSpecifications(final Specifications value)
    {
        this.specifications = value;
    }

    /**
     * Gets the implementations of the module.
     *
     * @return implementations of the module.
     */
    public Implementations getImplementations()
    {
        if(this.implementations == null)
        {
            this.implementations = new Implementations();
        }

        return this.implementations;
    }

    /**
     * Setter for property {@code implementations}.
     *
     * @param value the new implementations of the module.
     */
    public void setImplementations(final Implementations value)
    {
        this.implementations = value;
    }

    /**
     * Gets the properties of the module.
     *
     * @return the properties of the module.
     */
    public Properties getProperties()
    {
        if(this.properties == null)
        {
            this.properties = new Properties();
        }

        return this.properties;
    }

    /**
     * Setter for property {@code properties}.
     *
     * @param value the new properties of the module.
     */
    public void setProperties(final Properties value)
    {
        this.properties = value;
    }

    /**
     * Gets the description of the module.
     *
     * @return the description of the module.
     */
    public String getDescription()
    {
        if(this.description == null)
        {
            this.description = "";
        }

        return this.description;
    }

    /**
     * Setter for property {@code description}.
     *
     * @param value the new description of the module.
     */
    public void setDescription(final String value)
    {
        this.description = value;
    }

    /**
     * Gets the name of the module.
     *
     * @return the unique name of the module.
     */
    public String getName()
    {
        if(this.name == null)
        {
            this.name = "";
        }

        return this.name;
    }

    /**
     * Setter for property {@code name}.
     *
     * @param value the new name of the module.
     */
    public void setName(final String value)
    {
        this.name = value;
    }

    /**
     * Gets the version of the module.
     *
     * @return the version of the module.
     */
    public String getVersion()
    {
        if(this.version == null)
        {
            this.version = "";
        }

        return this.version;
    }

    /**
     * Setter for property {@code version}.
     *
     * @param value the new version of the module.
     */
    public void setVersion(final String value)
    {
        this.version = value;
    }

    /**
     * Creates a string representing the properties of the instance.
     *
     * @return a string representing the properties of the instance.
     */
    private String internalString()
    {
        return new StringBuffer(500).
            append("\n\tdescription=").append(this.description).
            append("\n\timplementations=").append(this.implementations).
            append("\n\tname=").append(this.name).
            append("\n\tproperties=").append(this.properties).
            append("\n\tspecifications=").append(this.specifications).
            append("\n\tversion=").append(this.version).
            toString();

    }

    //------------------------------------------------------------------Module--
    //--Object------------------------------------------------------------------

    /**
     * Returns a string representation of the object.
     *
     * @return a string representation of the object.
     */
    public String toString()
    {
        return super.toString() + this.internalString();
    }

    /**
     * Creates and returns a deep copy of this object.
     *
     * @return a clone of this instance.
     */
    public Object clone()
    {
        try
        {
            final Module ret = (Module) super.clone();
            ret.properties = (Properties) this.getProperties().clone();
            ret.implementations =
                (Implementations) this.getImplementations().clone();

            ret.specifications =
                (Specifications) this.getSpecifications().clone();

            return ret;
        }
        catch(CloneNotSupportedException e)
        {
            throw new AssertionError(e);
        }
    }

    /**
     * Indicates whether some other object is equal to this one by comparing
     * property {@code name}.
     *
     * @param o the reference object with which to compare.
     *
     * @return {@code true} if this object is the same as {@code o};
     * {@code false} otherwise.
     */
    public final boolean equals(final Object o)
    {
        return o == this || (o != null && o instanceof Module &&
            ((Module)o).getName().equals(this.getName()));

    }

    /**
     * Returns a hash code value for this object.
     *
     * @return a hash code value for this object.
     */
    public final int hashCode()
    {
        return this.getName().hashCode();
    }

    //------------------------------------------------------------------Object--

}
