/*
 *  jDTAUS - DTAUS fileformat.
 *  Copyright (c) 2005 Christian Schulte <cs@schulte.it>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */
package org.jdtaus.core.container;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;

/**
 * Specification meta-data.
 * <p>A specification consists of the properties {@code identifier},
 * {@code vendor}, {@code description} and {@code version}. Property
 * {@code identifier} holds an identifier uniquely identifying the specification
 * in a collection of specifications. Property {@code vendor} holds vendor
 * information for the vendor providing the specification. Property
 * {@code description} holds a textual description and property {@code version}
 * holds a textual version of the specification. The {@code singleton} flag
 * indicates whether implementations of the specification should be instantiated
 * once or whenever requested through the container.</p>
 *
 * @author <a href="mailto:cs@schulte.it">Christian Schulte</a>
 * @version $Id: Specification.java 2230 2007-03-26 01:37:48Z schulte2005 $
 */
public class Specification implements Cloneable, Serializable
{

    //--Specification-----------------------------------------------------------

    /**
     * The name of the module holding the specification.
     * @serial
     */
    private String moduleName;

    /**
     * The description of the specification.
     * @serial
     */
    private String description;

    /**
     * The identifier of the specification.
     * @serial
     */
    private String identifier;

    /**
     * The flag indicating that instance of the specification should be
     * created using a singleton strategy.
     * @serial
     */
    private boolean singleton;

    /**
     * The vendor of the specification.
     * @serial
     */
    private String vendor;

    /**
     * The version of the specification.
     * @serial
     */
    private String version;

    /**
     * The implementations available for the specification.
     * @serial
     */
    private Implementations implementations;

    /**
     * Maps implementation names to implementations.
     * @serial
     */
    private final Map implementationNames = new HashMap();

    /**
     * Gets the name of the module holding the specification.
     *
     * @return the name of the module holding the specification.
     */
    public String getModuleName()
    {
        if(this.moduleName == null)
        {
            this.moduleName = "";
        }

        return this.moduleName;
    }

    /**
     * Setter for property {@code moduleName}.
     *
     * @param value the new name of the module holding the specification.
     */
    public void setModuleName(final String value)
    {
        this.moduleName = value;
    }

    /**
     * Gets the description of the specification.
     *
     * @return the description of the specification.
     */
    public String getDescription()
    {
        if(this.description == null)
        {
            this.description = "";
        }

        return this.description;
    }

    /**
     * Setter for property {@code description}.
     *
     * @param value the new description of the specification.
     */
    public void setDescription(final String value)
    {
        this.description = value;
    }

    /**
     * Gets the identifier of the specification.
     *
     * @return the unique identifier of the specification.
     */
    public String getIdentifier()
    {
        if(this.identifier == null)
        {
            this.identifier = "";
        }

        return this.identifier;
    }

    /**
     * Setter for property {@code identifier}.
     *
     * @param value the new identifier of the specification.
     */
    public void setIdentifier(final String value)
    {
        this.identifier = value;
    }

    /**
     * Gets the flag indicating the instantiation strategy of the specification.
     *
     * @return {@code true} if the specification is specifying a singleton;
     * {@code false} if not.
     */
    public boolean isSingleton()
    {
        return this.singleton;
    }

    /**
     * Setter for property {@code singleton}.
     *
     * @param value {@code true} to flag the specification as a singleton;
     * {@code false} to not flag the specification as a singleton.
     */
    public void setSingleton(boolean value)
    {
        this.singleton = value;
    }

    /**
     * Gets the vendor of the specification.
     *
     * @return the vendor of the specification.
     */
    public String getVendor()
    {
        if(this.vendor == null)
        {
            this.vendor = "";
        }

        return this.vendor;
    }

    /**
     * Setter for property {@code vendor}.
     *
     * @param value the new vendor of the specification.
     */
    public void setVendor(final String value)
    {
        this.vendor = value;
    }

    /**
     * Gets the version of the specification.
     *
     * @return the version of the specification.
     */
    public String getVersion()
    {
        if(this.version == null)
        {
            this.version = "";
        }

        return this.version;
    }

    /**
     * Setter for property {@code version}.
     *
     * @param value the new version of the specification.
     */
    public void setVersion(final String value)
    {
        this.version = value;
    }

    /**
     * Gets an implementation for a name.
     *
     * @param name the name of the implementation to return.
     *
     * @return a reference to the implementation named {@code name}.
     *
     * @throws NullPointerException if {@code name} is {@code null}.
     * @throws MissingImplementationException if no implementation matching
     * {@code name} exists.
     */
    public Implementation getImplementation(final String name)
    {
        if(name == null)
        {
            throw new NullPointerException("name");
        }

        final Implementation ret =
            (Implementation) this.implementationNames.get(name);

        if(ret == null)
        {
            throw new MissingImplementationException(name);
        }

        return ret;
    }

    /**
     * Gets all available implementations of the specification.
     *
     * @return all available implementations of the specification.
     */
    public Implementations getImplementations()
    {
        if(this.implementations == null)
        {
            this.implementations = new Implementations();
        }

        return this.implementations;
    }

    /**
     * Setter for property {@code implementations}.
     *
     * @param value the new implementations of the specification.
     *
     * @throws DuplicateImplementationException if {@code value} contains
     * duplicate implementations.
     */
    public void setImplementations(final Implementations value)
    {
        if(value != null)
        {
            this.implementationNames.clear();
            for(int i = value.size() - 1; i >= 0; i--)
            {
                if(this.implementationNames.put(
                    value.getImplementation(i).getName(),
                    value.getImplementation(i)) != null)
                {

                    throw new DuplicateImplementationException(
                        value.getImplementation(i).getName());

                }
            }
        }
        this.implementations = value;
    }

    /**
     * Creates a string representing the properties of the instance.
     *
     * @return a string representing the properties of the instance.
     */
    private String internalString()
    {
        return new StringBuffer(500).
            append("\n\tdescription=").append(this.description).
            append("\n\tidentifier=").append(this.identifier).
            append("\n\timplementations=").append(this.implementations).
            append("\n\tmoduleName=").append(this.moduleName).
            append("\n\tsingleton=").append(this.singleton).
            append("\n\tvendor=").append(this.vendor).
            append("\n\tversion=").append(this.version).
            toString();

    }

    //-----------------------------------------------------------Specification--
    //--Object------------------------------------------------------------------

    /**
     * Returns a string representation of the object.
     *
     * @return a string representation of the object.
     */
    public String toString()
    {
        return super.toString() + this.internalString();
    }

    /**
     * Creates and returns a deep copy of this object.
     *
     * @return a clone of this instance.
     */
    public Object clone()
    {
        try
        {
            final Specification ret = (Specification) super.clone();
            ret.setImplementations(
                (Implementations) this.getImplementations().clone());

            return ret;
        }
        catch(CloneNotSupportedException e)
        {
            throw new AssertionError(e);
        }
    }

    /**
     * Indicates whether some other object is equal to this one by comparing
     * property {@code identifier}.
     *
     * @param o the reference object with which to compare.
     *
     * @return {@code true} if this object is the same as {@code o};
     * {@code false} otherwise.
     */
    public final boolean equals(final Object o)
    {
        return o == this || (o != null && o instanceof Specification &&
            ((Specification)o).getIdentifier().equals(this.getIdentifier()));

    }

    /**
     * Returns a hash code value for this object.
     *
     * @return a hash code value for this object.
     */
    public final int hashCode()
    {
        return this.getIdentifier().hashCode();
    }

    //------------------------------------------------------------------Object--

}
