/*
 *  jDTAUS - DTAUS fileformat.
 *  Copyright (c) 2005 Christian Schulte <cs@schulte.it>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */
package org.jdtaus.core.monitor;

import java.io.Serializable;
import java.rmi.server.UID;
import org.jdtaus.core.text.Message;

/**
 * A task of execution.
 * <p>A task is a sequence of operations taking time. If property
 * {@code indeterminate} is {@code false}, properties {@code minimum},
 * {@code maximum} and {@code progress} hold information regarding the progress
 * of the task.</p>
 *
 * @author <a href="mailto:cs@schulte.it">Christian Schulte</a>
 * @version $Id: Task.java 2916 2007-04-20 19:43:03Z schulte2005 $
 *
 * @see TaskEventSource
 */
public abstract class Task implements Cloneable, Serializable
{

    //--Constructors------------------------------------------------------------

    /** Creates a new {@code Task} instance. */
    public Task()
    {
        super();
        this.uid = new UID();
        this.indeterminate = true;
    }

    //------------------------------------------------------------Constructors--
    //--Task--------------------------------------------------------------------

    /**
     * Unique task identifier.
     * @serial
     */
    private UID uid;

    /**
     * The lower bound of the range.
     * @serial
     */
    private int minimum;

    /**
     * The upper bound of the range.
     * @serial
     */
    private int maximum;

    /**
     * Indicates the progress of the task.
     * @serial
     */
    private int progress;

    /**
     * Flag indicating if the operations performed by the task are of unknown
     * length.
     * @serial
     */
    private boolean indeterminate;

    /**
     * Gets the lower end of the progress value.
     *
     * @return an int representing the minimum value.
     */
    public final int getMinimum()
    {
        return this.minimum;
    }

    /**
     * Sets the lower end of the progress value.
     *
     * @param minimum an int specifying the minimum value.
     */
    public final void setMinimum(int minimum)
    {
        this.minimum = minimum;
    }

    /**
     * Gets the higher end of the progress value.
     *
     * @return an int representing the maximum value.
     */
    public final int getMaximum()
    {
        return this.maximum;
    }

    /**
     * Sets the higher end of the progress value.
     *
     * @param maximum an int specifying the maximum value.
     */
    public final void setMaximum(int maximum)
    {
        this.maximum = maximum;
    }

    /**
     * Gets the progress of the task.
     *
     * @return the progress of the task.
     */
    public final int getProgress()
    {
        return this.progress;
    }

    /**
     * Sets the progress of the task.
     *
     * @param progress an int specifying the current value, between the
     * maximum and minimum specified for this task.
     *
     * @throws IllegalArgumentException if {@code progress} is lower than
     * the minimum of the range or greater than the maximum of the range.
     */
    public final void setProgress(final int progress)
    {
        if(progress < this.minimum || progress > this.maximum)
        {
            throw new IllegalArgumentException(Integer.toString(progress));
        }

        this.progress = progress;
    }

    /**
     * Flag indicating if the operations performed by the task are of unknown
     * length.
     *
     * @return {@code true} if the operations performed by the task are of
     * unknown length; {@code false} if properties {@code minimum},
     * {@code maximum} and {@code progress} hold progress information.
     */
    public final boolean isIndeterminate()
    {
        return this.indeterminate;
    }

    /**
     * Setter for property {@code indeterminate}.
     *
     * @param indeterminate {@code true} if the operations performed by the task
     * are of unknown length; {@code false} if properties {@code minimum},
     * {@code maximum} and {@code progress} hold progress information.
     */
    public final void setIndeterminate(final boolean indeterminate)
    {
        this.indeterminate = indeterminate;
    }

    /**
     * Getter for property {@code description}.
     *
     * @return description of the task.
     */
    public abstract Message getDescription();

    /**
     * Creates a string representing the properties of the instance.
     *
     * @return a string representing the properties of the instance.
     */
    private String internalString()
    {
        return new StringBuffer(500).
            append("\n\tindeterminate=").append(this.indeterminate).
            append("\n\tmaximum=").append(this.maximum).
            append("\n\tminimum=").append(this.minimum).
            append("\n\tprogress=").append(this.progress).
            append("\n\tuid=").append(this.uid).
            append("\n\tdescription=").append(this.getDescription()).
            toString();

    }

    //--------------------------------------------------------------------Task--
    //--Object------------------------------------------------------------------

    /**
     * Returns a string representation of the object.
     *
     * @return a string representation of the object.
     */
    public String toString()
    {
        return super.toString() + this.internalString();
    }

    /**
     * Returns a hash code value for this object.
     *
     * @return a hash code value for this object.
     */
    public final int hashCode()
    {
        return this.uid.hashCode();
    }

    /**
     * Indicates whether some other object is equal to this one.
     * <p>Tasks internally cary a UID which is created during instantiation.
     * This UID is used for comparing {@code o} with the instance.</p>
     *
     * @param o the reference object with which to compare.
     *
     * @return {@code true} if this object is the same as {@code o};
     * {@code false} otherwise.
     *
     * @see UID
     */
    public final boolean equals(final Object o)
    {
        return o == this || (o instanceof Task &&
            ((Task) o).uid.equals(this.uid));

    }

    /**
     * Creates and returns a copy of this object.
     *
     * @return a clone of this instance.
     */
    public Object clone()
    {
        try
        {
            return super.clone();
        }
        catch(CloneNotSupportedException e)
        {
            throw new AssertionError(e);
        }
    }

    //------------------------------------------------------------------Object--

}
