/*
 *  jDTAUS - DTAUS fileformat.
 *  Copyright (c) 2005 Christian Schulte <cs@schulte.it>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */
package org.jdtaus.core.text;

import java.io.Serializable;
import java.rmi.server.UID;
import java.util.Comparator;
import java.util.Locale;

/**
 * Application message.
 * <p>Application messages consist of at least the two properties
 * {@code timestamp} and {@code formatArguments}. The {@code timestamp} property
 * will be initialized during instantiation to hold the timestamp of instance
 * creation. Property {@code formatArguments} holds the arguments to use
 * for formatting message text. It is recommended that subclasses of this class
 * are declared {@code final} so that calling {@code getClass()} on a message
 * instance always returns a type uniquely identifying a message type in the
 * system.</p>
 *
 * @author <a href="mailto:cs@schulte.it">Christian Schulte</a>
 * @version $Id: Message.java 2201 2007-03-21 23:59:00Z schulte2005 $
 *
 * @see MessageEventSource
 */
public abstract class Message implements Cloneable, Serializable
{

    //--Constants---------------------------------------------------------------

    /** Comparator for sorting messsages in ascending order of occurence. */
    public static final Comparator ASCENDING = new Comparator()
    {
        public int compare(final Object o1, final Object o2)
        {
            if(!(o1 instanceof Message) || !(o2 instanceof Message))
            {
                throw new IllegalArgumentException();
            }

            // TODO JDK 1.5 Long.valueOf(long)
            final Long l1 = new Long(((Message) o1).getTimestamp());
            final Long l2 = new Long(((Message) o2).getTimestamp());

            return l1.compareTo(l2);
        }
    };

    /** Comparator for sorting messsages in descending order of occurence. */
    public static final Comparator DESCENDING = new Comparator()
    {
        public int compare(final Object o1, final Object o2)
        {
            if(!(o1 instanceof Message) || !(o2 instanceof Message))
            {
                throw new IllegalArgumentException();
            }

            // TODO JDK 1.5 Long.valueOf(long)
            final Long l1 = new Long(((Message) o1).getTimestamp());
            final Long l2 = new Long(((Message) o2).getTimestamp());

            return l2.compareTo(l1);
        }
    };

    //---------------------------------------------------------------Constants--
    //--Constructors------------------------------------------------------------

    /** Creates a new {@code Message} instance. */
    public Message()
    {
        super();
        this.timestamp = System.currentTimeMillis();
        this.uid = new UID();
    }

    //------------------------------------------------------------Constructors--
    //--Message-----------------------------------------------------------------

    /**
     * Unique message identifier.
     * @serial
     */
    private UID uid;

    /**
     * The timestamp this message got created.
     * @serial
     */
    private long timestamp;

    /**
     * Getter for property {@code timestamp}.
     *
     * @return the timestamp this message got created.
     */
    public final long getTimestamp()
    {
        return this.timestamp;
    }

    /**
     * Getter for property {@code formatArguments}.
     *
     * @param locale the locale to be used for the arguments to return.
     *
     * @return the arguments to use when formatting the message text.
     */
    public abstract Object[] getFormatArguments(Locale locale);

    /**
     * Gets the formatted message text.
     *
     * @param locale the locale to be used for the text to return.
     */
    public abstract String getText(Locale locale);

    /**
     * Creates a string representing the properties of the instance.
     *
     * @return a string representing the properties of the instance.
     */
    private String internalString()
    {
        return new StringBuffer(500).
            append("\n\ttimestamp=").append(this.timestamp).
            append("\n\tuid=").append(this.uid).
            append("\n\tformatArguments=").
            append(this.getFormatArguments(Locale.getDefault())).
            append("\n\ttext=").append(this.getText(Locale.getDefault())).
            toString();

    }

    //-----------------------------------------------------------------Message--
    //--Object------------------------------------------------------------------

    /**
     * Returns a string representation of the object.
     *
     * @return a string representation of the object.
     */
    public String toString()
    {
        return super.toString() + this.internalString();
    }

    /**
     * Returns a hash code value for this object.
     *
     * @return a hash code value for this object.
     */
    public final int hashCode()
    {
        return this.uid.hashCode();
    }

    /**
     * Indicates whether some other object is equal to this one.
     * <p>Messages internally cary a UID which is created during instantiation.
     * This UID is used for comparing {@code o} with the instance.</p>
     *
     * @param o the reference object with which to compare.
     *
     * @return {@code true} if this object is the same as {@code o};
     * {@code false} otherwise.
     *
     * @see UID
     */
    public final boolean equals(final Object o)
    {
        return o == this || (o instanceof Message &&
            ((Message) o).uid.equals(this.uid));

    }

    /**
     * Creates and returns a copy of this object.
     *
     * @return a clone of this instance.
     */
    public Object clone()
    {
        try
        {
            return super.clone();
        }
        catch(CloneNotSupportedException e)
        {
            throw new AssertionError(e);
        }
    }

    //------------------------------------------------------------------Object--

}
