/*
 *  jDTAUS - DTAUS fileformat.
 *  Copyright (c) 2005 Christian Schulte <cs@schulte.it>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */
package org.jdtaus.core.text;

import java.util.EventObject;

/**
 * Event holding a message.
 *
 * @author <a href="mailto:cs@schulte.it">Christian Schulte</a>
 * @version $Id: MessageEvent.java 2201 2007-03-21 23:59:00Z schulte2005 $
 */
public class MessageEvent extends EventObject
{

    //--Constants---------------------------------------------------------------

    /** Event constant for information messages. */
    public static final int INFORMATION = 1;

    /** Event constant for notification messages. */
    public static final int NOTIFICATION = 2;

    /** Event constant for warning messages. */
    public static final int WARNING = 3;

    //---------------------------------------------------------------Constants--
    //--Constructors------------------------------------------------------------

    /**
     * Creates a new instance of {@code MessageEvent}.
     *
     * @param source the source of the new event.
     * @param message the message causing the new event.
     * @param type constant indicating the type of the event.
     *
     * @throws IllegalArgumentException if {@code type} is not equal to
     * one of the constants {@code INFORMATION}, {@code NOTIFICATION} and
     * {@code WARNING}.
     */
    public MessageEvent(final Object source, final Message message,
        final int type)
    {

        super(source);
        this.message = message;
        this.type = type;

        if(type != MessageEvent.INFORMATION
            && type != MessageEvent.NOTIFICATION
            && type != MessageEvent.WARNING)
        {

            throw new IllegalArgumentException(Integer.toString(type));
        }
    }

    //------------------------------------------------------------Constructors--
    //--MessageEvent------------------------------------------------------------

    /**
     * Message causing this event.
     * @serial
     */
    private final Message message;

    /**
     * Type of the event.
     * @serial
     */
    private final int type;

    /**
     * Getter for property {@code message}.
     *
     * @return the message causing this event.
     */
    public final Message getMessage()
    {
        return this.message;
    }

    /**
     * Getter for property {@code type}.
     *
     * @return the type of the event.
     */
    public final int getType()
    {
        return this.type;
    }

    /**
     * Creates a string representing the properties of the instance.
     *
     * @return a string representing the properties of the instance.
     */
    private String internalString()
    {
        return new StringBuffer(500).
            append("\n\ttype=").append(this.type).
            append("\n\tmessage=").append(this.message).
            toString();

    }

    //------------------------------------------------------------MessageEvent--
    //--Object------------------------------------------------------------------

    /**
     * Returns a string representation of the object.
     *
     * @return a string representation of the object.
     */
    public String toString()
    {
        return super.toString() + this.internalString();
    }

    //------------------------------------------------------------------Object--

}
