/*
 * Copyright 2000-2020 JetBrains s.r.o. and other contributors. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
 */

package com.jetbrains.plugin.structure.dotnet

import com.jetbrains.plugin.structure.base.decompress.DecompressorSizeLimitExceededException
import com.jetbrains.plugin.structure.base.plugin.*
import com.jetbrains.plugin.structure.base.problems.*
import com.jetbrains.plugin.structure.base.utils.*
import com.jetbrains.plugin.structure.dotnet.beans.ReSharperPluginBean
import com.jetbrains.plugin.structure.dotnet.beans.ReSharperPluginBeanExtractor
import com.jetbrains.plugin.structure.dotnet.problems.createIncorrectDotNetPluginFileProblem
import org.slf4j.LoggerFactory
import org.xml.sax.SAXParseException
import java.nio.file.Files
import java.nio.file.Path
import java.nio.file.Paths

@Suppress("unused")
class ReSharperPluginManager private constructor(private val extractDirectory: Path) : PluginManager<ReSharperPlugin> {
  companion object {
    private val LOG = LoggerFactory.getLogger(ReSharperPluginManager::class.java)

    fun createManager(
      extractDirectory: Path = Paths.get(Settings.EXTRACT_DIRECTORY.get())
    ): ReSharperPluginManager {
      extractDirectory.createDir()
      return ReSharperPluginManager(extractDirectory)
    }
  }

  override fun createPlugin(pluginFile: Path): PluginCreationResult<ReSharperPlugin> {
    require(pluginFile.exists()) { "Plugin file $pluginFile does not exist" }
    return when (pluginFile.extension) {
      "nupkg" -> loadDescriptorFromNuPkg(pluginFile)
      else -> PluginCreationFail(createIncorrectDotNetPluginFileProblem(pluginFile.simpleName))
    }
  }

  private fun loadDescriptorFromNuPkg(pluginFile: Path): PluginCreationResult<ReSharperPlugin> {
    val sizeLimit = Settings.RE_SHARPER_PLUGIN_SIZE_LIMIT.getAsLong()
    if (Files.size(pluginFile) > sizeLimit) {
      return PluginCreationFail(PluginFileSizeIsTooLarge(sizeLimit))
    }
    val tempDirectory = Files.createTempDirectory(extractDirectory, "plugin_")
    return try {
      val extractedDirectory = tempDirectory.resolve("content")
      extractZip(pluginFile, extractedDirectory, sizeLimit)
      loadDescriptorFromDirectory(extractedDirectory)
    } catch (e: DecompressorSizeLimitExceededException) {
      return PluginCreationFail(PluginFileSizeIsTooLarge(e.sizeLimit))
    } catch (e: Exception) {
      return PluginCreationFail(UnableToExtractZip())
    } finally {
      tempDirectory.deleteLogged()
    }
  }

  private fun loadDescriptorFromDirectory(pluginDirectory: Path): PluginCreationResult<ReSharperPlugin> {
    val candidateDescriptors = pluginDirectory.listRecursivelyAllFilesWithExtension("nuspec")
    if (candidateDescriptors.isEmpty()) {
      return PluginCreationFail(PluginDescriptorIsNotFound("*.nuspec"))
    }
    val descriptorFile = candidateDescriptors.first()
    if (candidateDescriptors.size > 1) {
      return PluginCreationFail(
        MultiplePluginDescriptors(
          descriptorFile.fileName.toString(),
          "plugin.nupkg",
          candidateDescriptors[1].fileName.toString(),
          "plugin.nupkg"
        )
      )
    }

    return loadDescriptor(descriptorFile)
  }

  private fun loadDescriptor(descriptorFile: Path): PluginCreationResult<ReSharperPlugin> {
    try {
      val descriptorContent = Files.readAllBytes(descriptorFile)
      val bean = ReSharperPluginBeanExtractor.extractPluginBean(descriptorContent.inputStream())
      val beanValidationResult = validateDotNetPluginBean(bean)
      if (beanValidationResult.any { it.level == PluginProblem.Level.ERROR }) {
        return PluginCreationFail(beanValidationResult)
      }
      val plugin = createPluginFromValidBean(bean, descriptorContent)
      return PluginCreationSuccess(plugin, beanValidationResult)
    } catch (e: SAXParseException) {
      val lineNumber = e.lineNumber
      val message = if (lineNumber != -1) "unexpected element on line $lineNumber" else "unexpected elements"
      return PluginCreationFail(UnexpectedDescriptorElements(message))
    } catch (e: Exception) {
      e.rethrowIfInterrupted()
      LOG.info("Unable to read plugin descriptor: ${descriptorFile.fileName}", e)
      return PluginCreationFail(UnableToReadDescriptor(descriptorFile.fileName.toString(), e.localizedMessage))
    }
  }

  private fun createPluginFromValidBean(bean: ReSharperPluginBean, nuspecFileContent: ByteArray) = with(bean) {
    val id = this.id!!
    val idParts = id.split('.')
    val vendor = if (idParts.size > 1) idParts[0] else null
    val authors = authors!!.split(',').map { it.trim() }
    val pluginName = when {
      title != null -> title!!
      idParts.size > 1 -> idParts[1]
      else -> id
    }
    ReSharperPlugin(
      pluginId = id, pluginName = pluginName, vendor = vendor, nonNormalizedVersion = this.version!!, url = this.url,
      changeNotes = this.changeNotes, description = this.description, vendorEmail = null, vendorUrl = null,
      authors = authors, licenseUrl = licenseUrl, copyright = copyright, summary = summary,
      dependencies = getAllDependencies().map { DotNetDependency(it.id!!, it.version) },
      nuspecFileContent = nuspecFileContent
    )
  }
}