/*
 * Copyright (c) 2013, Pavel Lechev
 *    All rights reserved.
 *
 *    Redistribution and use in source and binary forms, with or without modification,
 *    are permitted provided that the following conditions are met:
 *
 *     1) Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
 *     2) Redistributions in binary form must reproduce the above copyright notice,
 *        this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
 *     3) Neither the name of the Pavel Lechev nor the names of its contributors may be used to endorse or promote
 *        products derived from this software without specific prior written permission.
 *
 *    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 *    INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *    IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *    (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 *    HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 *    ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.jmockring;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.mockito.internal.matchers.NotNull.NOT_NULL;
import static org.mockito.internal.matchers.Null.NULL;

import org.hamcrest.CoreMatchers;
import org.jboss.resteasy.client.ClientResponse;
import org.jboss.resteasy.client.core.executors.URLConnectionClientExecutor;
import org.jmockring.annotation.ExecutionConfiguration;
import org.jmockring.annotation.RemoteRequestListener;
import org.jmockring.annotation.Server;
import org.jmockring.annotation.WebContext;
import org.jmockring.configuration.ServerExecutionConfiguration;
import org.jmockring.junit.ExternalServerJUnitRunner;
import org.jmockring.webserver.JettyWebServerImpl;
import org.jmockring.webserver.callback.CallbackRequestEventListener;
import org.jmockring.webserver.callback.Request;
import org.junit.Ignore;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.RestTemplate;

/**
 * @author Pavel Lechev
 * @date 20/07/12
 */
@RunWith(ExternalServerJUnitRunner.class)
@Server(
        name = "WebAppContextIT",
        webContexts = {
                @WebContext(
                        webApp = "/src/main/resources/webapp",
                        descriptor = "/src/main/resources/webapp/WEB-INF/web.xml",
                        contextPath = "/context1"
                )
                ,
                @WebContext(
                        webApp = "/src/main/resources/webapp",
                        descriptor = "/src/main/resources/webapp/WEB-INF/web.xml",
                        contextPath = "/context2"
                )
        },
        propertiesLocation = "/ri/base-webserver-ri.properties",
        bootstrap = JettyWebServerImpl.class
)
@Ignore("@WebContext needs more work")
public class WebAppContextIT {

    @ExecutionConfiguration(executionName = "WebAppContextIT", contextPath = "/context1")
    private ServerExecutionConfiguration config1;

    @ExecutionConfiguration(executionName = "WebAppContextIT", contextPath = "/context2")
    private ServerExecutionConfiguration config2;

    @RemoteRequestListener(executionName = "WebAppContextIT", contextPath = "/context1")
    private CallbackRequestEventListener requestListener;

    private RestTemplate restTemplate = new RestTemplate();

    @Test
    public void shouldAccessExternalContext1() throws Exception {
        ResponseEntity<String> result = restTemplate.getForEntity(config1.getRequestURL("/default-servlet"), String.class);

        assertThat(result.getStatusCode(), CoreMatchers.is(HttpStatus.OK));
        Object body = result.getBody();
        assertThat(body.toString(), is("[GET OK] URI=/context1/default-servlet"));
    }

    @Test
    public void shouldAccessExternalContext2() throws Exception {
        ResponseEntity<String> result = restTemplate.getForEntity(config2.getRequestURL("/default-servlet"), String.class);

        assertThat(result.getStatusCode(), CoreMatchers.is(HttpStatus.OK));
        Object body = result.getBody();
        assertThat(body.toString(), is("[GET OK] URI=/context2/default-servlet"));
    }


    @Test
    public void shouldWaitForRemoteRequestStart() throws Exception {

        new Thread() {
            @Override
            public void run() {
                try {
                    Thread.sleep(2000);
                    restTemplate.getForEntity(config1.getRequestURL("/default-servlet?test=START"), String.class);
                } catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
            }
        }.start();

        Request remReq = requestListener.waitForRequestStart(5);

        assertThat(remReq, NOT_NULL);
        assertThat(remReq.getMethod(), CoreMatchers.is(Request.Method.GET));
        assertThat(remReq.getRequestUri(), is("/context1/default-servlet"));
        assertThat(remReq.getParameters().get("test"), is(new String[]{"START"}));
    }

    @Test
    public void shouldWaitForRemoteRequestEnd() throws Exception {

        new Thread() {
            @Override
            public void run() {
                try {
                    Thread.sleep(2000);
                    restTemplate.getForEntity(config1.getRequestURL("/default-servlet?test=END"), String.class);
                } catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
            }
        }.start();

        Request remReq = requestListener.waitForRequestEnd(5);

        assertThat(remReq, NOT_NULL);
        assertThat(remReq.getMethod(), CoreMatchers.is(Request.Method.GET));
        assertThat(remReq.getRequestUri(), is("/context1/default-servlet"));
        assertThat(remReq.getParameters().get("test"), is(new String[]{"END"}));
    }

    @Test
    public void shouldWaitForRemoteRequestStartAndEnd() throws Exception {

        new Thread() {
            @Override
            public void run() {
                try {
                    Thread.sleep(2000);
                    restTemplate.getForEntity(config1.getRequestURL("/default-servlet?test=START-END"), String.class);
                } catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
            }
        }.start();

        Request remReqStart = requestListener.waitForRequestStart(5);
        Request remReqEnd = requestListener.waitForRequestEnd(5);

        assertThat(remReqStart, NOT_NULL);
        assertThat(remReqStart.getMethod(), CoreMatchers.is(Request.Method.GET));
        assertThat(remReqStart.getRequestUri(), is("/context1/default-servlet"));
        assertThat(remReqStart.getParameters().get("test"), is(new String[]{"START-END"}));

        assertThat(remReqEnd, NOT_NULL);
        assertThat(remReqEnd.getMethod(), CoreMatchers.is(Request.Method.GET));
        assertThat(remReqEnd.getRequestUri(), is("/context1/default-servlet"));
        assertThat(remReqEnd.getParameters().get("test"), is(new String[]{"START-END"}));
    }


    @Test
    public void shouldWaitAndReturnNullForRemoteRequestTimeout() throws Exception {

        new Thread() {
            @Override
            public void run() {
                try {
                    Thread.sleep(7000); // call it is 7 seconds, but our timeout is only 5 sec, so should time out and get NULL
                    restTemplate.getForEntity(config1.getRequestURL("/default-servlet?test=START"), String.class);
                } catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
            }
        }.start();

        Request remReq = requestListener.waitForRequestStart(5);
        assertThat(remReq, NULL);

        Thread.sleep(3000);  // allow some time for the spawned thread to connect (else the server will shut down)
    }


    @Test
    @Ignore("Bootstrapping Resteasy is broken")
    public void shouldAccessExternalRestEndpoint() throws Exception {
        ClientResponse result = new URLConnectionClientExecutor().createRequest(config2.getRequestURL("/default/111")).get();
        assertThat(result.getStatus(), is(200));
        Object body = result.getEntity(String.class);
        assertThat(body.toString(), is("[GET OK] URI=/context2/default-servlet"));
    }


}
