/**
 *                                  Apache License
 *                            Version 2.0, January 2004
 *                         http://www.apache.org/licenses/
 *
 *    TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION
 *
 *    1. Definitions.
 *
 *       "License" shall mean the terms and conditions for use, reproduction,
 *       and distribution as defined by Sections 1 through 9 of this document.
 *
 *       "Licensor" shall mean the copyright owner or entity authorized by
 *       the copyright owner that is granting the License.
 *
 *       "Legal Entity" shall mean the union of the acting entity and all
 *       other entities that control, are controlled by, or are under common
 *       control with that entity. For the purposes of this definition,
 *       "control" means (i) the power, direct or indirect, to cause the
 *       direction or management of such entity, whether by contract or
 *       otherwise, or (ii) ownership of fifty percent (50%) or more of the
 *       outstanding shares, or (iii) beneficial ownership of such entity.
 *
 *       "You" (or "Your") shall mean an individual or Legal Entity
 *       exercising permissions granted by this License.
 *
 *       "Source" form shall mean the preferred form for making modifications,
 *       including but not limited to software source code, documentation
 *       source, and configuration files.
 *
 *       "Object" form shall mean any form resulting from mechanical
 *       transformation or translation of a Source form, including but
 *       not limited to compiled object code, generated documentation,
 *       and conversions to other media types.
 *
 *       "Work" shall mean the work of authorship, whether in Source or
 *       Object form, made available under the License, as indicated by a
 *       copyright notice that is included in or attached to the work
 *       (an example is provided in the Appendix below).
 *
 *       "Derivative Works" shall mean any work, whether in Source or Object
 *       form, that is based on (or derived from) the Work and for which the
 *       editorial revisions, annotations, elaborations, or other modifications
 *       represent, as a whole, an original work of authorship. For the purposes
 *       of this License, Derivative Works shall not include works that remain
 *       separable from, or merely link (or bind by name) to the interfaces of,
 *       the Work and Derivative Works thereof.
 *
 *       "Contribution" shall mean any work of authorship, including
 *       the original version of the Work and any modifications or additions
 *       to that Work or Derivative Works thereof, that is intentionally
 *       submitted to Licensor for inclusion in the Work by the copyright owner
 *       or by an individual or Legal Entity authorized to submit on behalf of
 *       the copyright owner. For the purposes of this definition, "submitted"
 *       means any form of electronic, verbal, or written communication sent
 *       to the Licensor or its representatives, including but not limited to
 *       communication on electronic mailing lists, source code control systems,
 *       and issue tracking systems that are managed by, or on behalf of, the
 *       Licensor for the purpose of discussing and improving the Work, but
 *       excluding communication that is conspicuously marked or otherwise
 *       designated in writing by the copyright owner as "Not a Contribution."
 *
 *       "Contributor" shall mean Licensor and any individual or Legal Entity
 *       on behalf of whom a Contribution has been received by Licensor and
 *       subsequently incorporated within the Work.
 *
 *    2. Grant of Copyright License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       copyright license to reproduce, prepare Derivative Works of,
 *       publicly display, publicly perform, sublicense, and distribute the
 *       Work and such Derivative Works in Source or Object form.
 *
 *    3. Grant of Patent License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       (except as stated in this section) patent license to make, have made,
 *       use, offer to sell, sell, import, and otherwise transfer the Work,
 *       where such license applies only to those patent claims licensable
 *       by such Contributor that are necessarily infringed by their
 *       Contribution(s) alone or by combination of their Contribution(s)
 *       with the Work to which such Contribution(s) was submitted. If You
 *       institute patent litigation against any entity (including a
 *       cross-claim or counterclaim in a lawsuit) alleging that the Work
 *       or a Contribution incorporated within the Work constitutes direct
 *       or contributory patent infringement, then any patent licenses
 *       granted to You under this License for that Work shall terminate
 *       as of the date such litigation is filed.
 *
 *    4. Redistribution. You may reproduce and distribute copies of the
 *       Work or Derivative Works thereof in any medium, with or without
 *       modifications, and in Source or Object form, provided that You
 *       meet the following conditions:
 *
 *       (a) You must give any other recipients of the Work or
 *           Derivative Works a copy of this License; and
 *
 *       (b) You must cause any modified files to carry prominent notices
 *           stating that You changed the files; and
 *
 *       (c) You must retain, in the Source form of any Derivative Works
 *           that You distribute, all copyright, patent, trademark, and
 *           attribution notices from the Source form of the Work,
 *           excluding those notices that do not pertain to any part of
 *           the Derivative Works; and
 *
 *       (d) If the Work includes a "NOTICE" text file as part of its
 *           distribution, then any Derivative Works that You distribute must
 *           include a readable copy of the attribution notices contained
 *           within such NOTICE file, excluding those notices that do not
 *           pertain to any part of the Derivative Works, in at least one
 *           of the following places: within a NOTICE text file distributed
 *           as part of the Derivative Works; within the Source form or
 *           documentation, if provided along with the Derivative Works; or,
 *           within a display generated by the Derivative Works, if and
 *           wherever such third-party notices normally appear. The contents
 *           of the NOTICE file are for informational purposes only and
 *           do not modify the License. You may add Your own attribution
 *           notices within Derivative Works that You distribute, alongside
 *           or as an addendum to the NOTICE text from the Work, provided
 *           that such additional attribution notices cannot be construed
 *           as modifying the License.
 *
 *       You may add Your own copyright statement to Your modifications and
 *       may provide additional or different license terms and conditions
 *       for use, reproduction, or distribution of Your modifications, or
 *       for any such Derivative Works as a whole, provided Your use,
 *       reproduction, and distribution of the Work otherwise complies with
 *       the conditions stated in this License.
 *
 *    5. Submission of Contributions. Unless You explicitly state otherwise,
 *       any Contribution intentionally submitted for inclusion in the Work
 *       by You to the Licensor shall be under the terms and conditions of
 *       this License, without any additional terms or conditions.
 *       Notwithstanding the above, nothing herein shall supersede or modify
 *       the terms of any separate license agreement you may have executed
 *       with Licensor regarding such Contributions.
 *
 *    6. Trademarks. This License does not grant permission to use the trade
 *       names, trademarks, service marks, or product names of the Licensor,
 *       except as required for reasonable and customary use in describing the
 *       origin of the Work and reproducing the content of the NOTICE file.
 *
 *    7. Disclaimer of Warranty. Unless required by applicable law or
 *       agreed to in writing, Licensor provides the Work (and each
 *       Contributor provides its Contributions) on an "AS IS" BASIS,
 *       WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *       implied, including, without limitation, any warranties or conditions
 *       of TITLE, NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A
 *       PARTICULAR PURPOSE. You are solely responsible for determining the
 *       appropriateness of using or redistributing the Work and assume any
 *       risks associated with Your exercise of permissions under this License.
 *
 *    8. Limitation of Liability. In no event and under no legal theory,
 *       whether in tort (including negligence), contract, or otherwise,
 *       unless required by applicable law (such as deliberate and grossly
 *       negligent acts) or agreed to in writing, shall any Contributor be
 *       liable to You for damages, including any direct, indirect, special,
 *       incidental, or consequential damages of any character arising as a
 *       result of this License or out of the use or inability to use the
 *       Work (including but not limited to damages for loss of goodwill,
 *       work stoppage, computer failure or malfunction, or any and all
 *       other commercial damages or losses), even if such Contributor
 *       has been advised of the possibility of such damages.
 *
 *    9. Accepting Warranty or Additional Liability. While redistributing
 *       the Work or Derivative Works thereof, You may choose to offer,
 *       and charge a fee for, acceptance of support, warranty, indemnity,
 *       or other liability obligations and/or rights consistent with this
 *       License. However, in accepting such obligations, You may act only
 *       on Your own behalf and on Your sole responsibility, not on behalf
 *       of any other Contributor, and only if You agree to indemnify,
 *       defend, and hold each Contributor harmless for any liability
 *       incurred by, or claims asserted against, such Contributor by reason
 *       of your accepting any such warranty or additional liability.
 *
 *    END OF TERMS AND CONDITIONS
 *
 *    APPENDIX: How to apply the Apache License to your work.
 *
 *       To apply the Apache License to your work, attach the following
 *       boilerplate notice, with the fields enclosed by brackets "{}"
 *       replaced with your own identifying information. (Don't include
 *       the brackets!)  The text should be enclosed in the appropriate
 *       comment syntax for the file format. We also recommend that a
 *       file or class name and description of purpose be included on the
 *       same "printed page" as the copyright notice for easier
 *       identification within third-party archives.
 *
 *    Copyright 2014 Edgar Espina
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */
package org.jooby.mongodb;

import static java.util.Objects.requireNonNull;

import java.util.function.BiConsumer;
import java.util.function.Consumer;

import org.jooby.Env;
import org.jooby.Env.ServiceKey;
import org.jooby.internal.mongodb.AutoIncID;
import org.jooby.internal.mongodb.GuiceObjectFactory;
import org.mongodb.morphia.Datastore;
import org.mongodb.morphia.EntityInterceptor;
import org.mongodb.morphia.Morphia;
import org.mongodb.morphia.annotations.PrePersist;
import org.mongodb.morphia.mapping.Mapper;

import com.google.inject.Binder;
import com.typesafe.config.Config;

/**
 * Extends {@link Mongodb} with object-document mapping via {@link Morphia}.
 *
 * Exposes {@link Morphia} and {@link Datastore} services.
 *
 * <h1>usage</h1>
 *
 * <p>
 * application.conf:
 * </p>
 *
 * <pre>
 * db = "mongodb://localhost/mydb"
 * </pre>
 *
 * <pre>
 * {
 *   use(new Monphia());
 *
 *   get("/", req {@literal ->} {
 *     Datastore ds = req.require(Datastore.class);
 *     // work with mydb datastore
 *   });
 * }
 * </pre>
 *
 * <h1>options</h1>
 * <h2>morphia callback</h2>
 * <p>
 * The {@link Morphia} callback let you map classes and/or set mapper options.
 * </p>
 *
 * <pre>
 * {
 *   use(new Monphia()
 *     .doWith((morphia, config) {@literal ->} {
 *       // work with morphia
 *       morphia.map(MyObject.class);
 *     });
 *   );
 * }
 * </pre>
 *
 * For more detailed information, check
 * <a href="https://github.com/mongodb/morphia/wiki/MappingObjects">here</a>
 *
 * <h2>datastore callback</h2>
 * <p>
 * This {@link Datastore} callback is executed only once, it's perfect for checking indexes:
 * </p>
 *
 * <pre>
 * {
 *   use(new Monphia()
 *     .doWith(datastore {@literal ->} {
 *       // work with datastore
 *       datastore.ensureIndexes();
 *       datastore.ensureCap();
 *     });
 *   );
 * }
 * </pre>
 *
 * For more detailed information, check
 * <a href="https://github.com/mongodb/morphia/wiki/Datastore#ensure-indexes-and-caps">here</a>
 *
 * <h2>auto-incremental ID</h2>
 * <p>
 * This modules comes with auto-incremental ID generation.
 * </p>
 * <p>
 * usage:
 * </p>
 * <pre>
 * {
 *   use(new Monphia().with(IdGen.GLOBAL); // or IdGen.LOCAL
 * }
 * </pre>
 * <p>
 * ID must be of type: {@link Long} and annotated with {@link GeneratedValue}:
 * </p>
 * <pre>
 * &#64;Entity
 * public class MyEntity {
 *   &#64;Id &#64;GeneratedValue Long id;
 * }
 * </pre>
 *
 * <p>
 * There two ID gen:
 * </p>
 * <ul>
 * <li>GLOBAL: generates a global and unique ID regardless of entity type.</li>
 * <li>LOCAL: generates an unique ID per entity type.</li>
 * </ul>
 *
 * <h1>entity listeners</h1>
 *
 * <p>
 * Guice will create and inject entity listeners (when need it).
 * </p>
 *
 * <pre>
 * public class MyListener {
 *
 *   private Service service;
 *
 *   &#64;Inject
 *   public MyListener(Service service) {
 *     this.service = service;
 *   }
 *
 *   &#64;PreLoad void preLoad(MyObject object) {
 *     service.doSomething(object);
 *   }
 *
 * }
 * </pre>
 *
 * <p>
 * NOTE: ONLY Constructor injection is supported.
 * </p>
 *
 * @author edgar
 * @since 0.5.0
 */
public class Monphia extends Mongodb {

  private BiConsumer<Morphia, Config> morphiaCbck;

  private Consumer<Datastore> callback;

  private IdGen gen = null;

  /**
   * Creates a new {@link Monphia} module.
   *
   * @param db Name of the property with the connection URI.
   */
  public Monphia(final String db) {
    super(db);
  }

  /**
   * Creates a new {@link Monphia} using the default property: <code>db</code>.
   */
  public Monphia() {
  }

  /**
   * Morphia startup callback, from here you can map classes, set mapper options, etc..
   *
   * @param callback Morphia callback.
   * @return This module.
   */
  public Monphia doWith(final BiConsumer<Morphia, Config> callback) {
    this.morphiaCbck = requireNonNull(callback, "Mapper callback is required.");
    return this;
  }

  /**
   * {@link Datastore} startup callback, from here you can call {@link Datastore#ensureIndexes()}.
   *
   * @param callback Datastore callback.
   * @return This module.
   */
  public Monphia doWith(final Consumer<Datastore> callback) {
    this.callback = requireNonNull(callback, "Datastore callback is required.");
    return this;
  }

  /**
   * <p>
   * Setup up an {@link EntityInterceptor} on {@link PrePersist} events that generates an
   * incremental ID.
   * </p>
   *
   * Usage:
   * <pre>
   * {
   *   use(new Monphia().with(IdGen.GLOBAL);
   * }
   * </pre>
   *
   * <p>
   * ID must be of type: {@link Long} and annotated with {@link GeneratedValue}:
   * </p>
   * <pre>
   * &#64;Entity
   * public class MyEntity {
   *   &#64;Id &#64;GeneratedValue Long id;
   * }
   * </pre>
   *
   * @param gen an {@link IdGen} strategy
   * @return This module.
   */
  public Monphia with(final IdGen gen) {
    this.gen = requireNonNull(gen, "ID Gen is required.");
    return this;
  }

  @Override
  public void configure(final Env env, final Config conf, final Binder binder) {
    configure(env, conf, binder, (uri, client) -> {
      String db = uri.getDatabase();

      Mapper mapper = new Mapper();

      Morphia morphia = new Morphia(mapper);

      if (this.morphiaCbck != null) {
        this.morphiaCbck.accept(morphia, conf);
      }

      Datastore datastore = morphia.createDatastore(client, mapper, db);
      if (gen != null) {
        mapper.addInterceptor(new AutoIncID(datastore, gen));
      }
      if (callback != null) {
        callback.accept(datastore);
      }

      ServiceKey serviceKey = env.serviceKey();
      serviceKey.generate(Morphia.class, db,
          k -> binder.bind(k).toInstance(morphia));
      serviceKey.generate(Datastore.class, db,
          k -> binder.bind(k).toInstance(datastore));

      env.onStart(registry -> new GuiceObjectFactory(registry, morphia));
    });
  }

}
